<?php

namespace App\Console\Commands;

use App\AppUserAuthorization;
use App\Content;
use App\ContentFiles;
use App\ContentListener;
use App\User;
use App\UserCredits;
use Carbon\Carbon;
use Exception;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Kreait\Firebase\Factory;
use Kreait\Firebase\Messaging\CloudMessage;
use Kreait\Firebase\Messaging\Notification;
class SendContinuePodcastNotification extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'app:send-continue-podcast-notification';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'An alert to let users know that they need to continue to listen to an episode to complete their CLE.';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $users = User::where('user_type_id', 3)->whereHas('contentListerners', function ($query) {
            $query->where('listen_status', 0);
        })->active()->get();
        
        foreach ($users as $user) {
            $userbars = getBarsOfUser($user->id);
            $bar_id = "";
            // Get shortest required length in seconds
            foreach ($userbars as $state) {
                $user_bar_subscription= $user->getActiveSubscription($state->bar_id);
                $bar_id = $user_bar_subscription->bar_id;
                if($user_bar_subscription){
                    $barCreditsSum= $user_bar_subscription->hours_needed_per_cycle ?? 0;
                }else{
                    $barCreditsSum= 0;
                }
            }
        
            $userContentListerners = $user->contentListerners()
                ->joinSub(
                    ContentListener::selectRaw('content_file_id, MAX(id) as max_id')
                        ->where('user_id', $user->id)
                        ->where('listen_status', 0)
                        ->whereBetween('created_at', [
                            Carbon::yesterday()->startOfDay(),
                            Carbon::yesterday()->endOfDay()
                        ])
                        ->groupBy('content_file_id'),
                    'latest',
                    function ($join) {
                        $join->on('content_listeners.id', '=', 'latest.max_id');
                    }
                )
                ->get();
                
            foreach ($userContentListerners as $content) {
                $podcast = ContentFiles::where('id', $content->content_file_id)->first();
                $credit_provided_by_the_course = $podcast->contentStates->where('bar_id', $bar_id)->firstOrFail()->credit_provided_by_the_course;
                $credits_provided_by_the_podcast = $this->calculateCredit($podcast->length, $credit_provided_by_the_course);
                $credits_earned_by_the_user_for_the_video = $this->calculateCredit(($podcast->getLastListenTimeInSecondsForUser($user->id)), $credit_provided_by_the_course);
                
                if (!$podcast) {
                    continue; // Skip if video file is not found
                }
        
                $listen_time = $content->listen_time_in_seconds;

                // Send notification if listen time is less than the required time
                if ($credits_earned_by_the_user_for_the_video < $credits_provided_by_the_podcast) {
                    // Log::info('send Notification');
                    // Log::info('$listen_time',[$listen_time]);
                    // Log::info('credits_provided_by_the_podcast', [$credits_provided_by_the_podcast]);
                    // Log::info('credits_earned_by_the_user_for_the_video', [$credits_earned_by_the_user_for_the_video]);

                    $this->sendPushNotification($user, ['file_name'=>$podcast->file_name, 'podcast_id'=>$podcast->contents_id, 'episode_id'=>$podcast->id]);
                }
            }
        }
    }
    
    protected function calculateCredit($timeInSeconds, $credit_provided_by_the_course)
    {
        $pc_mins = secondsToMinute($timeInSeconds);

        if (!isset($credit_provided_by_the_course)) {
            $credit_provided_by_the_course = 0; // or some other default value
        }
        $credit =$credit_provided_by_the_course != 0 ? floor($pc_mins / $credit_provided_by_the_course) * 0.25 : 0;
        return $credit;
    }

    private function sendPushNotification($user, $data)
    {
            $deviceToken = AppUserAuthorization::where('user_id',$user->id)->latest()->first();

            $file_name = $data['file_name'];

            // foreach($deviceToken as $token){
            if($deviceToken != null){
                if($deviceToken->device_token){
                $message = "Continue listening to your episode, “ $file_name “ on CLEarly Legal to complete your CLE credits";
                $this->sendFCMNotification($deviceToken->device_token, $message, $data);
            }else{
                Log::warning('Device token invalid or not found');
            }  
            } 
        // }   
    }
    
    private function sendFCMNotification($deviceToken, $message, $data)
    {
        try {
            $messaging = app('firebase.messaging');
    
            $podcast_id = $data['podcast_id'];
            $episode_id = $data['episode_id'];
            $message = CloudMessage::withTarget('token', $deviceToken)
                ->withNotification(Notification::create('CLE Reminder', $message))
                ->withData(['episode_id' => $episode_id, 'podcast_id' =>$podcast_id, 'action' => "resume_podcast"]);
    
            $messaging->send($message);
            Log::info("Notification sent successfully.", ['data' => $message ]);
        } catch (Exception $e) {
            Log::error("FCM Error", ['Exception' => $e->getMessage()]);
        }
    }
    
         
}
