<?php

namespace App\Console\Commands;

use App\ContentFiles;
use App\ContentListener;
use App\User;
use App\Submissions;
use App\Notifications\SendWeeklySummaryReport;
use App\SiteConfig;
use App\Subscription;
use App\UserCredits;
use Illuminate\Console\Command;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class WeeklySummaryEmail extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'send:weekly-summary-email';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Send weekly summary report via email';

    /**
     * Execute the console command.
     */
    public function handle()
    {

        $accessToken = SiteConfig::where('name', 'mail_access_token')->first();
        if (isset($accessToken)) {
            config([
                'mail.from.address' => siteconfig('from_email'),
                'mail.from.name' => siteconfig('from_name'),
                'mail.password' => $accessToken->val,
            ]);
        }

        
        // Get all users with active subscriptions
        $subscriptions = Subscription::with(['user' => function ($query) {
            $query->where('status', 1);
        }])->where('stripe_subscription_status', 'active')->where('status', 1)->get();
    
        foreach ($subscriptions as $subscription) {
            $user = $subscription->user;
    
            $podcastsCL = [];
            $podcastYetToSubmit = [];
            $credits_earned = 0;
            $podcasts_completed_listening = [];
    
            if ($user) {
                $userbars = $user ?? getBarsOfUser($user->id);
                $bar_id = "";
                // Get shortest required length in seconds
                foreach ($userbars as $state) {
                    $user_bar_subscription= $subscription;
                    $bar_id = $subscription->bar_id;
                    if($user_bar_subscription){
                        $barCreditsSum= $user_bar_subscription->hours_needed_per_cycle ?? 0;
                    }else{
                        $barCreditsSum= 0;
                    }
                }
                // Define date range for the last week starting from Monday
                $lastMonday = Carbon::now()->startOfWeek(Carbon::MONDAY)->subWeek();
                // $lastSunday = Carbon::now()->startOfWeek(Carbon::MONDAY)->subDay();
                $lastSunday = Carbon::now();
    
                // Podcasts completed and listened to within the last week
                $podcasts_listening = ContentListener::where('user_id', $user->id)
                    ->groupBy('content_file_id')
                    ->whereBetween('updated_at', [$lastMonday, $lastSunday])
                    ->get();
    
                foreach ($podcasts_listening as $content) {
                    $podcast = ContentFiles::where('id', $content->content_file_id)->first();
                    $credit_provided_by_the_course = $podcast->contentStates->where('bar_id', $bar_id)->firstOrFail()->credit_provided_by_the_course;
                    $credits_provided_by_the_podcast = $this->calculateCredit($podcast->length, $credit_provided_by_the_course);
                    $credits_earned_by_the_user_for_the_video = $this->calculateCredit(($podcast->getLastListenTimeInSecondsForUser($user->id)), $credit_provided_by_the_course);
                    
                    if (!$podcast) {
                        continue; // Skip if video file is not found
                    }
            
                    $listen_time = $content->listen_time_in_seconds;

                    // Send notification if listen time is less than the required time
                    if ($credits_earned_by_the_user_for_the_video >= $credits_provided_by_the_podcast) {
                        $podcasts_completed_listening[] = $podcast;
                        
                    }
                }
                foreach ($podcasts_completed_listening as $pcl) {
                    // $contentFile = ContentFiles::where('id', $pcl->content_file_id)
                    //     ->where('contents_id', $pcl->content_id)
                    //     ->first();
                    // if ($contentFile) {
                    //     $podcastsCL[] = $podcasts_completed_listening;
                    // }

                    $podcastsCL = $podcasts_completed_listening;
                    
                    // Podcasts completed but not yet submitted
                    $podcasts_completed_but_yet_to_submit = UserCredits::where('user_id', $user->id)
                        ->where('submission_id', 0)
                        ->where('content_file_id', $pcl->id)
                        ->whereBetween('created_at', [$lastMonday, $lastSunday])
                        ->groupBy('content_file_id')
                        ->get();
    
                    foreach ($podcasts_completed_but_yet_to_submit as $pclys) {
                        $contentFileToBeSubmitted = ContentFiles::where('id', $pclys->content_file_id)->first();
                        if ($contentFileToBeSubmitted) {
                            $podcastYetToSubmit[] = $contentFileToBeSubmitted;
                        }
                    }
    
                    // Calculate credits earned within the last week
                    $credits_earned += UserCredits::where('user_id', $user->id)
                        ->where('submission_id', 0)
                        ->where('content_file_id', $pcl->id)
                        ->whereBetween('created_at', [$lastMonday, $lastSunday])
                        ->sum('credit');
                }
            }
            $latestPodcasts = ContentFiles::orderBy('created_at', 'desc')
                        ->take(5)
                        ->get();
    
            // Send email notification
            if (!empty($user->email)) {
                $user->notify(new SendWeeklySummaryReport($podcastsCL, $podcastYetToSubmit, $credits_earned, $latestPodcasts));
            }
        }
    }

    protected function calculateCredit($timeInSeconds, $credit_provided_by_the_course)
    {
        $pc_mins = secondsToMinute($timeInSeconds);

        if (!isset($credit_provided_by_the_course)) {
            $credit_provided_by_the_course = 0; // or some other default value
        }
        $credit =$credit_provided_by_the_course != 0 ? floor($pc_mins / $credit_provided_by_the_course) * 0.25 : 0;
        return $credit;
    }
}
