<?php

namespace App;

use App\Model;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Log;

use function PHPUnit\Framework\isNull;

class Content extends Model
{
    /**
     * Overwrite created_by field value with currently logged in user.
     * Set @var has_created_by to false if created_by field does not exist in DB Table.
     *
     * @var boolean
     */
    protected $has_created_by = true;

    public $user;

    protected static function boot()
    {
        parent::boot();
        static::retrieved(function ($content) {
            if (Auth::check()) {
                $user = Auth::user();
                $content->setUser($user);
            }
            // else {
            //     Log::info("Auth::check() is false Model:Content");
            // }
        });
    }

    public function setUser($user)
    {
        $this->user = $user;
    }

    /**
     * Overwrite updated_by field value with currently logged in user.
     * Set @var has_updated_by to false if created_by field does not exist in DB Table.
     *
     * @var boolean
     */

    protected $has_updated_by = true;

    /**
     * Define feilds name which have html tags
     * Set @var notStripTags add DB Table column name which column have html tags.
     *
     * @var array
     */

    public static $notStripTags = [];

    protected $dates = [
        'expire_date'
    ];

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'content_type',
        'category_id',
        'podcast_title',
        'title_crc32',
        'internal_cl_podcast_id',
        'short_description',
        'provider_type',
        'provider_id',
        'cl_type',
        'file_type',
        'prerequisites',
        'expire_date',
        'temp_expire_date',
        'image',
        'image_alt',
        'status',
        'published_on',
        'created_by',
        'updated_by'
    ];


    protected $appends = ['image_path', 'provider_name', 'view_counts', 'temp_expire_date'];

    public function getExpireDateAttribute($value)
    {
        if(!isNull($value)){
            return Carbon::parse($value)->format(config('app.db_date_format'));
        }else{
            return $value;
        }

    }

    public function getPublishedOnAttribute($value)
    {
        if(!isNull($value)){
            return Carbon::parse($value)->format(config('app.db_date_format'));
        }else{
            return $value;
        }

    }

    public function getApproveStateNameAttribute()
    {
        $contentsStates = $this->contentsStates;
        $state_name = [];
        if ($contentsStates) {
            foreach ($contentsStates as $state) {
                $state_name[] = $state->state_name;
            }
        }
        // Join the names using comma and return
        return implode(', ', $state_name);
    }

    /**
     * The get result based on search criteria.
     * @param  \Illuminate\Http\Request  $request
     * @return object App\Bar
     */
    public function getResult($request)
    {
        // Set default parameter values
        $order_by = !empty($request->get('order_by')) ? $request->get('order_by') : 'published_on';
        $order = !empty($request->get('order')) ? $request->get('order') : 'desc';

        // Fetch contents list
        $contents = new Content;

        // Search
        if (!empty($request->get('search'))) {
            $searchStr = $request->get('search');
            $escape = "ESCAPE '|'";
            if (substr_count($searchStr, "|")) {
                $searchStr = str_replace('\\', '\\\\\\', $searchStr);
                $escape = "";
            }
            // added escape for searching backslash issue DLC-140
            $contents = $contents->where(function ($query) use ($searchStr, $escape) {
                $query
                    ->whereRaw('podcast_title LIKE ?  ' . $escape, '%' . $searchStr . '%');
            });
        }

        // Status
        if ($request->get('status') !== null) {
            $contents = $contents->where('status', $request->get('status'));
        }



        // Category
        if ($request->get('category') !== null) {
            $contents = $contents->where('category_id', $request->get('category'));
        }

        // Approved States 
        if ($request->get('bar') !== null) {
            $bar = $request->get('bar');
            $contents = $contents->whereHas('contentsStates', function ($query) use ($bar) {
                $query->where('bar_id', $bar);
            });

        }

        // Approved States 
        if ($request->get('speaker') !== null) {
            $speaker = $request->get('speaker');
            $contents = $contents->whereHas('speakers', function ($query) use ($speaker) {
                $query->where('speaker_id', $speaker);
            });

        }
        //if per_content not set then all record value use for per_content
        // Curretly it is not in Admin user listing as we are using data table paging
        // but have kept this for API integration
        $per_page = !empty($request->get('per_page')) ? $request->get('per_page') : $contents->count();
        // Order By & Pagination
        $contents = $contents->orderBy($order_by, $order)->paginate($per_page);

        return $contents;
    }


    // public function getUploadFileAttribute($value)
    // {
    //     return json_decode($value, true);
    // }

    function createdByUser()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    function category()
    {
        return $this->belongsTo(Category::class, 'category_id');
    }

    public function getProviderNameAttribute()
    {
        if ($this->provider_type == 1) {
            $provider = User::where('id', $this->provider_id)->select('first_name','last_name')->first();
            return $provider ? ($provider->first_name.' '.$provider->last_name) : null;
        } else {
            $provider = Provider::where('id', $this->provider_id)->select('name')->first();
            return $provider ? $provider->name : null;
        }
    }




    public function getImagePathAttribute()
    {
        return getImageUrl($this->image, 'content');
    }

    public function contentsStates()
    {
        return $this->hasMany(ContentsStates::class, 'contents_id');
    }

    public function contentFiles()
    {
        return $this->hasMany(ContentFiles::class, 'contents_id')->where('length', '>', 0);
        //->select('id', 'contents_id', 'file', 'length');
    }

    public function lengthOfPodcast()
    {
        $contentFilesLengthSum = $files = $this->relationLoaded('contentFiles') 
        ? $this->getRelation('contentFiles') 
        : $this->contentFiles;
        $contentFilesLengthSum = $contentFilesLengthSum->sum('length');
        $pc_mins = secondsToMinute($contentFilesLengthSum);
        return $pc_mins;

    }

    public function sizeOfPodcast()
    {
        $contentFilesSizeSum = $files = $this->relationLoaded('contentFiles') 
        ? $this->getRelation('contentFiles') 
        : $this->contentFiles;
        $contentFilesSizeSum = $contentFilesSizeSum->sum('size');
        return $contentFilesSizeSum;

    }

    public function totalCreditsOfPodcast($barName)
    {
        $contentFiles= $files = $this->relationLoaded('contentFiles') 
        ? $this->getRelation('contentFiles') 
        : $this->contentFiles;
    
        $totalCredit = 0;
        foreach ($contentFiles as $contentFile) {
            $totalCredit += $contentFile->calculateCreditForBar($this->lengthOfPodcast(), $barName);
        }
    
    return $totalCredit;
    }

    public function contentListeners()
    {
        return $this->hasMany(ContentListener::class, 'content_id', 'id');
    }

    public function getUserCredits($id)
    {
        $userCredits = UserCredits::where('content_id', $id)->get();
        return $userCredits;
    }


    public function getViewCountsAttribute()
    {
        return $this->contentListeners()->count();
    }

    public function getPodcastStatusAttribute()
    {
        // 0 = pending
        // 1 = completed
        // 2 = submitted

        $podcast_status = 0;

        $statuses = $this->relationLoaded('contentFiles') ? $this->getRelation('contentFiles') : $this->contentFiles;
        $statuses = $statuses->pluck('status');
        if ($statuses->contains(0)) {
            return 0;
        } elseif (($statuses->contains(1) || ($statuses->contains(1) && $statuses->contains(2))) && !$statuses->contains(0) && !$statuses->contains(3) && !$statuses->contains(4) && !$statuses->contains(5)) {
            return 1;
        } elseif ($statuses->contains(2) && !$statuses->contains(1) && !$statuses->contains(0) && !$statuses->contains(3) && !$statuses->contains(4) && !$statuses->contains(5)) {
            return 2;
        } elseif ($statuses->contains(3) && !$statuses->contains(0) && !$statuses->contains(1) && !$statuses->contains(2) && !$statuses->contains(4) && !$statuses->contains(5) ) {
            return 3;
        }elseif ((!$statuses->contains(4) && ($statuses->contains(1) || $statuses->contains(2)  || $statuses->contains(3) || $statuses->contains(5))) || ($statuses->contains(4) && ($statuses->contains(1) || $statuses->contains(2)  || $statuses->contains(3) || $statuses->contains(4)|| $statuses->contains(5)))) {
            return 4;
        } elseif ($statuses->contains(5) && !$statuses->contains(0)) {
            return 5;
        }

        // $podcast_credits = $this->getPodcastCredits(1);
        // $podcast_earned_credits = $this->getPodcastCredits(2);
        // $podcast_submitted_credits = $this->getPodcastCredits(3);

        // if($podcast_earned_credits >= $podcast_credits){
        //     $podcast_status=1;
        //     if($podcast_submitted_credits == $podcast_credits){
        //         $podcast_status=2;
        //     }
        // }else{
        //     $podcast_status=0;
        // }

        return $podcast_status;
    }

    public function getTempExpireDateAttribute()
    {
        if (!is_null($this->expire_date)) {
            return getTempExpireDate($this->expire_date);
        } else {
            return null;
        }
    }

    public function speakers()
    {
        return $this->hasMany(PodcastRefSpeakers::class, 'podcast_id', 'id');
    }

    public function getSpeakersNameAttribute()
    {
        $speakers = $this->speakers;
        $speaker_name = [];
        if ($speakers) {
            foreach ($speakers as $speaker) {
                if($speaker->speaker->status == 1){
                    $speaker_name[] = $speaker->speaker->name;
                }
            }
        }
        // Join the names using comma and return
        return implode(', ', $speaker_name);
    }

}
