<?php

namespace App;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use App\Model;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use App\Bar;

class ContentFiles extends Model
{
    use HasFactory;

    protected $table = "content_upload";
    public $user;
    public $userProfile;

    protected static function boot()
    {
        parent::boot();
        static::retrieved(function ($contentFiles) {
            // if (app()->runningInConsole()) {
            //     return;
            // }
            if (Auth::check()) {
                $user = Auth::user();
                $contentFiles->setUser($user);
            } 
            // else {
            //     Log::info("Auth::check() is false Model:ContentFiles");
            // }
        });
    }

    public function setUser($user)
    {
        $this->user = $user;
        $this->setuserProfile($this->user->getProfile());
    }
    public function setuserProfile($data)
    {
        $this->userProfile = $data;
    }

    /**
     * Define feilds name which have html tags
     * Set @var notStripTags add DB Table column name which column have html tags.
     *
     * @var array
     */

    public static $notStripTags = [];


    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'contents_id',
        'view_file_name',
        'bar_id',
        'file',
        'length',
        'size',
    ];

    protected $appends = ['file_path', 'file_name', 'length_in_seconds', 'last_listen_time_in_seconds', 'view_counts', 'credit', 'credit_earned', 'status'];

    public function content()
    {
        return $this->belongsTo(Content::class, 'id', 'contents_id');
    }

    public function episodeStats()
    {
        $stats = PodcastStatistics::where('content_file_id', $this->id)->latest()->get();
        return $stats;
    }

    public function contentStates()
    {
        return $this->hasMany(ContentsStates::class, 'contents_id', 'contents_id')
                    ->whereColumn('bar_id', 'bar_id');
    }

    public function getFilePathAttribute()
    {
        return getImageUrl($this->file, 'upload_file');
    }

    public function getFileNameAttribute()
    {
        if($this->view_file_name == '' || $this->view_file_name == null){
            $lastUnderscorePosition = strrpos($this->file, '_');
            $lastUnderscorePositionOfExtension = strrpos($this->file, '.');
            return substr($this->file, 0, $lastUnderscorePosition) . '.' . substr($this->file, $lastUnderscorePositionOfExtension + 1);
        }else{
            return $this->view_file_name;
        }
    }

    public function getLengthInSecondsAttribute()
    {
        return minuteSecondToSeconds($this->length);
    }

    public function contentListeners()
    {
        return $this->hasMany(ContentListener::class, 'content_file_id', 'id');
    }

    public function getLastListenTimeInSecondsAttribute()
    {
        $listener = $this->contentListeners()->where('user_id', Auth::id())->latest()->first();
        return $listener ? $listener->listen_time : 0;
    }

    public function getLastListenTimeInSecondsForUser($user_id)
    {
        $listener = $this->contentListeners()->where('user_id', $user_id)->latest()->first();
        return $listener ? $listener->listen_time : 0;
    }

    /*
    public function getLastListenTimeInSecondsAttribute()
    {
        $listener = $this->belongsTo(ContentListener::class, 'id', 'content_file_id')->select('listen_time')->where('user_id', $this->user->id)->latest()->first();
        return $listener ? $listener->listen_time : null;
        // return $listener ? minuteSecondToSeconds($listener->listen_time) : null;
    }

    */

    public function getViewCountsAttribute()
    {
        return $this->contentListeners()->count();
    }

    public function getStatusAttribute()
    {
        // Initialize status
        $status = 0;

        // Get credits
        $pc_credit = $this->calculateCredit($this->length);
        $pc_earned_credit = $this->calculateCredit($this->last_listen_time_in_seconds);

        if($this->last_listen_time_in_seconds == 0){
            //not yet started
            $status = 3;
        }elseif($this->last_listen_time_in_seconds < $this->length && $this->last_listen_time_in_seconds > 0){
            //in progress
            $status = 4;
        }
        // Compare earned credit with total credit
        if ($pc_earned_credit >= $pc_credit) {
            $status = 1; // Completed by default

            // Get user profile once
            $userProfile = $this->userProfile;
            if ($userProfile) {
                $userId = $userProfile['id'];
                // Get bar ID
                if (isset($this->userProfile['statebars'][0])) {
                    $bar_id = $userProfile['statebars'][0]->bar_id;

                    // Check if submitted credit matches total credit
                        $submitted_credit_for_video = Submissions::where('content_id', $this->contents_id)
                        ->where('content_file_id', $this->id)
                        ->where('user_id', $userId)
                        ->where('bar_id', $bar_id)
                        ->sum('credit_submitted');

                    if ($submitted_credit_for_video == $this->credit) {
                        $status = 2; // Submitted
                    }
                }
            }
        }

        return $status;
    }


    protected function calculateCredit($timeInSeconds)
    {
        if (!$this->userProfile) {
            return 0;
        }

        $pc_mins = secondsToMinute($timeInSeconds);

        if (isset($this->userProfile['statebars'][0])) {
            // $barCreditsSum = $this->userProfile['statebars'][0]->barCreditsSum;
            $bar_id = $this->userProfile['statebars'][0]->bar_id;
            $credit_provided_by_the_course = $this->contentStates->where('bar_id', $bar_id)->firstOrFail()->credit_provided_by_the_course;

        } else {
            $credit_provided_by_the_course = 0; // or some other default value
        }
        $credit = $credit_provided_by_the_course != 0 ? floor($pc_mins / $credit_provided_by_the_course) * 0.25 : 0;
        return $credit;
    }

    protected function calculateCreditForWeb($barCreditsSum)
    {
        $credit =$barCreditsSum != 0 ? floor(secondsToMinute($this->length) / $barCreditsSum) * 0.25 : 0;
        return $credit;
    }

    public function getCreditAttribute()
    {
        return $this->calculateCredit($this->length);
    }

    public function getCreditEarnedAttribute()
    {
        return $this->calculateCredit($this->last_listen_time_in_seconds);
    }

    public function calculateCreditForBar($timeInMinutes, $barName)
    {
        $bar = Bar::where('state_name', $barName)->first();
        $hours_needed_per_cycle = $bar->hours_needed_per_cycle;
        return $this->calculateCreditForWeb($hours_needed_per_cycle);
    }

}
