<?php

namespace App\Http\Controllers\Admin;

use App\Faculty;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Content;
use App\User;
use App\State;
use App\Bar;
use App\Category;
use App\ContentFiles;
use App\ContentListener;
use App\ContentsStates;
use Illuminate\Validation\Rule;
use App\Notifications\UserVerification;
use App\Events\BulkAction;
use App\Notifications\ConnectedSpeakerToPodcastNotification;
use App\PodcastRefSpeakers;
use App\Provider;
use App\UserCredits;
use Carbon\Carbon;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Str;
use Svg\Tag\Rect;

class ContentController extends Controller
{
    protected $indexRoute = 'content.index';
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $content = new Content;
        $content = $content->getResult($request);
        $categories = Category::all();
        $bars = Bar::all();
        $speakers = Faculty::where('status', 1)->get();

        return view('admin.modules.content.index')->with(['contents'=> $content, 'categories' =>$categories, 'bars'=>$bars, 'speakers'=>$speakers]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $content = new Content;
        $data = $this->getContentPageData($content);
        $categories = Category::active()->get();
        $faculty = Faculty::active()->orderBy('name', 'asc')->get();
        $providersList = [];

        return view('admin.modules.content.addedit', compact('categories', 'providersList', 'faculty'))->with($data, $categories, $providersList, $faculty);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request, Content $content)
    {
        try{
            $request = $this->stripHtmlTags($request, Content::$notStripTags);
            $originalPublishedOn = $request['published_on'] ?? null;
            $originalExpireOn = $request['expire_date'] ?? null;
            
            try {
                if ($request['published_on']) {
                    $parsedPublished = Carbon::createFromFormat('m/d/Y H:i', $request['published_on']);
            
                    if ($parsedPublished->equalTo(Carbon::create(1899, 12, 31, 0, 0))) {
                        return redirect()->back()->withErrors(['published_on' => 'The published date format is not valid.']);
                    }
            
                    $request['published_on'] = $parsedPublished->format('Y-m-d');
                }
            } catch (\Exception $e) {
                return redirect()->back()->withErrors(['published_on' => 'The published date format is not valid.']);
            }
            
            try {
                if ($request['expire_date']) {
                    $request['expire_date'] = Carbon::createFromFormat('m/d/Y', $request['expire_date'])->format('Y-m-d');
                }
            } catch (\Exception $e) {
                return redirect()->back()->withErrors(['expire_date' => 'The expire date format is not valid.']);
            }
            
            
            $this->validation($request);
            
            $request['published_on'] = $originalPublishedOn;
            $request['expire_date'] = $originalExpireOn;
            
            $data = $request->all();
            $voluntary_content = 0;
            if (isset($data['voluntary_content'])) {
                $voluntary_content = $data['voluntary_content'];
            }
            $data['voluntary_content'] = $voluntary_content;
            $cle_mandatory = 0;
            if (isset($data['cle_mandatory'])) {
                $cle_mandatory = $data['cle_mandatory'];
            }
            $data['cle_mandatory'] = $cle_mandatory;
            $podcast_accepted = 0;
            if (isset($data['podcast_accepted'])) {
                $podcast_accepted = $data['podcast_accepted'];
            }
            $data['podcast_accepted'] = $podcast_accepted;
            $auto_credit_upload = 0;
            if (isset($data['auto_credit_upload'])) {
                $auto_credit_upload = $data['auto_credit_upload'];
            }
            $data['auto_credit_upload'] = $auto_credit_upload;

            // if($request->multiimage){
            //     $data['upload_file'] = json_encode($request->multiimage);
            // }
            // else{
            //     $data['upload_file'] = json_encode([$request->upload_file]);
            // }

            if($request->expire_date){
                $inputDate = Carbon::parse($request->expire_date);
                $data['expire_date'] = $inputDate;
                $data['temp_expire_date'] = getTempExpireDate($data['expire_date']);
            }else{
                $data['expire_date'] = Null;
            }

            if($request->published_on){
                $data['published_on'] = Carbon::parse($request->published_on);
            }else{
                $data['published_on'] = Null;
            }

            if (($data['provider_type'] == 2 || $data['provider_type'] == 3) && $data['provider_id'] == 'other') {
                $provider = new Provider();
                $provider->name = $data['provider_name'];
                $provider->user_type = ($data['provider_type'] == 2) ? 1 : 2;
                $provider->save();
                $data['provider_id'] = $provider->id;
            }
            if (!$data['image_alt']) {
                $data['image_alt'] = 'Clealry Content';
            }
            $content->fill($data);

            $latestContent = Content::latest()->first();
            $latestId = $latestContent ? $latestContent->id : null;
            $newId = 'CL-' . str_pad($latestId + 1, 4, '0', STR_PAD_LEFT);
            $content->internal_cl_podcast_id = $newId;

            $content->title_crc32 = crc32_encode($data['podcast_title']);

            $content->save();
            for ($i = 0; $i < count($data['state_id']); $i++) {
                ContentsStates::create([
                    'contents_id' => $content->id,
                    'bar_id' => $data['state'][$i],
                    'credit_provided_by_the_course' => $data['credit_provided_by_the_course'][$i],
                    'state_bar_id' => $data['state_bar_id'][$i]
                ]);
            }
            if (isset($request->multiimage)) {
                $rules = [
                    'multiimagefilename.*' => 'required',
                    'multiimagebarid.*' => 'required',
                ];
                $messages = [
                    'multiimagefilename.*.required' => 'The file name is required.',
                    'multiimagebarid.*.required' => 'The bar ID is rrquired.',
                ];

                $this->validate($request, $rules, $messages);
                for ($i = 0; $i < count($request->multiimage); $i++) {
                    ContentFiles::create([
                        'contents_id' => $content->id,
                        'view_file_name' => $request->multiimagefilename[$i],
                        'bar_id' => $request->multiimagebarid[$i],
                        'file' => $request->multiimage[$i],
                        'length' => minuteSecondToSeconds($request->multiimageduration[$i]),
                        'size' => bytesToMB($request->multiimagesize[$i])
                    ]);
                }
                Artisan::call('admin:add-content-file-size');
            }


            if (isset($request->speaker)){
                try {
                    PodcastRefSpeakers::where('podcast_id', $content->id)->delete();
                
                    foreach ($request->speaker as $speakerId) {
                        PodcastRefSpeakers::create([
                            'podcast_id' => $content->id,
                            'speaker_id' => $speakerId,
                        ]);
                    }
                
                    $speakers = $content->speakers()->get();
                    $firstFile = $content->contentFiles->first();
                    $contentLength = $firstFile ? $firstFile->length : 0;

                    $activeSpeakers = $speakers->filter(function ($speaker) {
                        return optional($speaker->speaker)->status == 1;
                    })->values();

                    $activeSpeakerNames = $activeSpeakers
                    ->map(fn($s) => optional($s->speaker)->name)
                    ->filter() // remove nulls
                    ->implode(', ');
                

                    Log::info($activeSpeakerNames);
                    foreach ($request->speaker as $speakerId) {
                        try {
                            $faculty = Faculty::findOrFail($speakerId);
                            Notification::route('mail', $faculty->email)
                                ->notify(new ConnectedSpeakerToPodcastNotification($faculty, $content, $activeSpeakerNames, $contentLength));
                        } catch (\Exception $e) {
                            Log::error("Notification failed for faculty ID: $speakerId", [
                                'error' => $e->getMessage(),
                            ]);
                            // Optionally continue with others
                            continue;
                        }
                    }
                } catch (\Exception $e) {
                    Log::error('Failed to assign speakers or send notifications.', [
                        'error' => $e->getMessage(),
                    ]);
                
                    return response()->json([
                        'success' => false,
                        'message' => 'An error occurred while assigning speakers or sending notifications.',
                    ], 500);
                }
                
            }

            if ($request->get('btnsave') == 'savecontinue') {
                return redirect()->route('content.edit', ['content' => $content->id])->with("success", __('content.create_success', ['name' => $request->get('name')]));
            } elseif ($request->get('btnsave') == 'save') {
                return redirect()->route($this->indexRoute)->with("success", __('content.create_success', ['name' => $request->get('name')]));
            } else {
                return redirect()->route($this->indexRoute);
            }
        } catch (\Throwable $e) {            
            $errors = method_exists($e, 'errors') ? $e->errors() : ['error' => $e->getMessage()];

            $flattenedErrors = [];
            
            foreach ($errors as $key => $value) {
                $flattenedErrors[$key] = is_array($value) ? implode(' ', $value) : $value;
            }
            return redirect()->back()->withErrors($flattenedErrors);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Content  $content
     * @return \Illuminate\Http\Response
     */
    public function show(Content $content)
    {
        $data = $this->getContentPageData($content);
        return view('admin.modules.content.view')->with($data);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Content  $content
     * @return \Illuminate\Http\Response
     */
    public function edit(Content $content, Request $request)
    {

        $data = $this->getContentPageData($content);
        $providersList = [];
        if ($data['content']['provider_type'] == 1) {
            $providersList = new User;
            $providersList = $providersList->getProviderAdminList();
        } else if ($data['content']['provider_type'] == 2) {
            $providersList = new Provider;
            $request->user_type = 1;
            $providersList = $providersList->getProvidersList($request);
        } else if ($data['content']['provider_type'] == 3) {
            $providersList = new Provider;
            $request->user_type = 2;
            $providersList = $providersList->getProvidersList($request);
        }

        $categories = Category::active()->get();
        $faculty = Faculty::active()->orderBy('name', 'asc')->get();
        $podcastSpeakers = $content->speakers()->get();

        return view('admin.modules.content.addedit', compact('categories', 'data', 'providersList','faculty', 'podcastSpeakers'))->with($data, $categories, $providersList, $faculty, $podcastSpeakers);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Content  $content
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, Content $content)
    {
        try{
            $request = $this->stripHtmlTags($request, Content::$notStripTags);
            $originalPublishedOn = $request['published_on'] ?? null;
            $originalExpireOn = $request['expire_date'] ?? null;
            
            try {
                if ($request['published_on']) {
                    $parsedPublished = Carbon::createFromFormat('m/d/Y H:i', $request['published_on']);
            
                    if ($parsedPublished->equalTo(Carbon::create(1899, 12, 31, 0, 0))) {
                        return redirect()->back()->withErrors(['published_on' => 'The published date format is not valid.']);
                    }
            
                    $request['published_on'] = $parsedPublished->format('Y-m-d');
                }
            } catch (\Exception $e) {
                return redirect()->back()->withErrors(['published_on' => 'The published date format is not valid.']);
            }
            
            try {
                if ($request['expire_date']) {
                    $request['expire_date'] = Carbon::createFromFormat('m/d/Y', $request['expire_date'])->format('Y-m-d');
                }
            } catch (\Exception $e) {
                return redirect()->back()->withErrors(['expire_date' => 'The expire date format is not valid.']);
            }
            
            
            $this->validation($request);
            
            $request['published_on'] = $originalPublishedOn;
            $request['expire_date'] = $originalExpireOn;
            // $this->validation($request, $content->id);
            $data = $request->all();
            $voluntary_content = 0;
            if (isset($data['voluntary_content'])) {
                $voluntary_content = $data['voluntary_content'];
            }
            $data['voluntary_content'] = $voluntary_content;
            $cle_mandatory = 0;
            if (isset($data['cle_mandatory'])) {
                $cle_mandatory = $data['cle_mandatory'];
            }
            $data['cle_mandatory'] = $cle_mandatory;
            $podcast_accepted = 0;
            if (isset($data['podcast_accepted'])) {
                $podcast_accepted = $data['podcast_accepted'];
            }
            $data['podcast_accepted'] = $podcast_accepted;
            $auto_credit_upload = 0;
            if (isset($data['auto_credit_upload'])) {
                $auto_credit_upload = $data['auto_credit_upload'];
            }
            $data['auto_credit_upload'] = $auto_credit_upload;
            $validate = 'no';
            if (isset($content->users) && $content->users->count() > 0 && $content->status == 1 && $data['status'] == 0) {
                $data['status'] = 1;
                $validate = 'yes';
            }

            if (isset($request->multiimage) && $content->upload_file != json_encode($request->multiimage)) {
                $data['upload_file'] = json_encode($request->multiimage);
            }
            if (isset($request->upload_file) && $content->upload_file != $request->upload_file) {
                $data['upload_file'] = json_encode([$request->upload_file]);
            }
            if($request->expire_date){
                $inputDate = Carbon::parse($request->expire_date);
                $data['expire_date'] = $inputDate;
                $data['temp_expire_date'] = getTempExpireDate($data['expire_date']);
            }else{
                $data['expire_date'] = Null;
            }

            if($request->published_on){
                $data['published_on'] = Carbon::parse($request->published_on);
            }else{
                $data['published_on'] = Null;
            }

            if (($data['provider_type'] == 2 || $data['provider_type'] == 3) && $data['provider_id'] == 'other') {
                $provider = new Provider();
                $provider->name = $data['provider_name'];
                $provider->user_type = ($data['provider_type'] == 2) ? 1 : 2;
                $provider->save();
                $data['provider_id'] = $provider->id;
            }
            if (!$data['image_alt']) {
                $data['image_alt'] = 'Clealry Content';
            }

            $content->title_crc32 = crc32_encode($data['podcast_title']);

            $content->fill($data);
            $content->save();

            $existingRecordsContentsStates = $content->contentsStates()->pluck('id')->toArray();
            for ($i = 0; $i < count($data['state_id']); $i++) {
                ContentsStates::updateOrCreate(
                    ['id' => $data['state_id'][$i]],
                    [
                        'contents_id' => $content->id,
                        'bar_id' => $data['state'][$i],
                        'credit_provided_by_the_course' => $data['credit_provided_by_the_course'][$i],
                        'state_bar_id' => $data['state_bar_id'][$i]
                    ]
                );
                $key = array_search($data['state_id'][$i], $existingRecordsContentsStates);
                unset($existingRecordsContentsStates[$key]);
            }
            ContentsStates::destroy($existingRecordsContentsStates);

            $existingRecordsContentFiles = $content->contentFiles()->pluck('id')->toArray();
            if($request->multiimage_ids){
                $rules = [
                    'multiimagefilename.*' => 'required',
                    'multiimagebarid.*' => 'required',
                ];
                $messages = [
                    'multiimagefilename.*.required' => 'The file name is required.',
                    'multiimagebarid.*.required' => 'The bar ID is rrquired.',
                ];

                $this->validate($request, $rules, $messages);
                for ($i = 0; $i < count($request->multiimage_ids); $i++) {

                    ContentFiles::updateOrCreate(
                        ['id' => $request->multiimage_ids[$i]],
                        [
                            'view_file_name' => $request->multiimagefilename[$i],
                            'bar_id' => $request->multiimagebarid[$i],
                            'contents_id' => $content->id,
                            'file' => $request->multiimage[$i],
                            'length' => minuteSecondToSeconds($request->multiimageduration[$i]),
                            'size' => bytesToMB($request->multiimagesize[$i])
                        ]
                    );
                    $key = array_search($request->multiimage_ids[$i], $existingRecordsContentFiles);
                    unset($existingRecordsContentFiles[$key]);
                }
                Artisan::call('admin:add-content-file-size');
            }
            ContentFiles::destroy($existingRecordsContentFiles);

            if ($validate == 'yes') {
                $message = __('content.update_success_validate', ['name' => $request->get('state_name')]);
            } else {
                $message = __('content.update_success', ['name' => $request->get('state_name')]);
            }

            if (isset($request->speaker)){
                PodcastRefSpeakers::where('podcast_id',$content->id)->delete();
                for ($i = 0; $i < count($request->speaker); $i++) {
                    PodcastRefSpeakers::create([
                        'podcast_id' => $content->id,
                        'speaker_id'=>$request->speaker[$i],
                    ]);
                }
            }


            if ($request->get('btnsave') == 'savecontinue') {
                return redirect()->route('content.edit', ['content' => $content->id])->with("success", $message);
            } elseif ($request->get('btnsave') == 'save') {
                return redirect()->route($this->indexRoute)->with("success", $message);
            } else {
                return redirect()->route($this->indexRoute);
            }
        } catch (\Throwable $e) {
            $errors = method_exists($e, 'errors') ? $e->errors() : ['error' => $e->getMessage()];

            $flattenedErrors = [];
            
            foreach ($errors as $key => $value) {
                $flattenedErrors[$key] = is_array($value) ? implode(' ', $value) : $value;
            }
            return redirect()->back()->withErrors($flattenedErrors);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Content  $content
     * @return \Illuminate\Http\Response
     */
    public function destroy(Content $content)
    {
        //
    }
    /**
     * Validate the Form
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    private function validation($request)
    {
        $rules = [
            //'content_type' => 'required',
            'podcast_title' => 'required',
            // 'internal_cl_podcast_id' => 'required',
            'category_id' => 'required',
            'provider_type' => 'required',
            'provider_id' => 'required',
            'state.*' => 'required',
            //'credit_provided_by_the_course.*' => 'required',
            'state_bar_id.*' => 'required',
            'file_type' => 'required',
            'cl_type' => 'required',
            'multiimagebarid.*' => 'required',
            'speaker' => 'required',
            'multiimagefilename.*' => 'required',
            // 'multiimage' => 'required',
            'provider_name' => [
                Rule::requiredIf(function () use ($request) {
                    return $request->provider_id == 'other';
                }),
            ],
            'credit_provided_by_the_course.*' => [
                'required', // Example: All fields are required by default
                function ($attribute, $value, $fail) use ($request) {
                    if (!empty($request->multiimageduration)) {
                        $fileDurations = array_map('minuteSecondToSeconds', $request->multiimageduration); // Convert all file durations to seconds
            
                        foreach ($request->credit_provided_by_the_course as $index => $requiredMinutes) {
                            $requiredSeconds = $requiredMinutes * 60; // Convert required minutes to seconds
                            
                            // Check if at least one file meets the required duration
                            $isValid = false;
                            foreach ($fileDurations as $fileDuration) {
                                if ($fileDuration >= $requiredSeconds) {
                                    $isValid = true;
                                    break;
                                }
                            }
            
                            // If no file meets the required duration, show validation error
                            if (!$isValid) {
                                $stateName = $request->state[$index] ?? 'Unknown State'; // Get the state name dynamically
                                $fail(__('content.this_state_credit_length_validation'));
                            }
                        }
                    }
                },
            ],
            'published_on' => [
                'nullable',
                'date',
                'before_or_equal:today',
                function ($attribute, $value, $fail) use ($request) {
                    if (!empty($request->expire_date) && $value >= $request->expire_date) {
                        $fail("The published on must be before the expire date.");
                    }
                },
            ],
            'expire_date' => 'nullable|date',

        ];

        // if($id > 0){
        //     $rules['cl_number'] = "required|unique:contents,cl_number,{$id},id,deleted_at,NULL";
        // }else{
        //     $rules['cl_number'] = ['required', Rule::unique('contents')->whereNull('deleted_at')];
        // }
        $messages = [
            'multiimage.required' => __('content.need_to_upload_file'),
            'category_id.required' => __('content.category_id_validate'),
            'provider_id.required' => __('content.provider_id_validate'),
            'state.*.required' => __('content.state_validate'),
            'credit_provided_by_the_course.*.required' => __('content.credit_provided_by_the_course_validate'),
            'state_bar_id.*.required' => __('content.state_bar_id_validate'),
            'multiimagefilename.*.required' => __('content.multiimagefilename_validate'),
            'multiimagebarid.*.required' => __('content.multiimagebarid_validate'),
            'podcast_title.required' => __('content.podcast_title_validate'),
            'published_on.before_or_equal' => 'The published date cannot be in the future.',
        ];

        $this->validate($request, $rules, $messages);
    }
    /**
     * Apply change visible
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function changestatus(Request $request)
    {
        $content = Content::findOrFail($request->id);
        if ($content->status  == 0) {
            Content::where('id', $request->id)->update(['status' => 1]);
            $message = __('content.record_active_success');
        } else {
            $validate = 'no';
            if (isset($content->users) && $content->users->count() > 0) {
                $validate = 'yes';
            } else {
                Content::where('id', $request->id)->update(['status' => 0]);
            }
            if ($validate == 'yes') {
                $message = __('content.record_inactive_success_validate');
            } else {
                $message = __('content.record_inactive_success');
            }
        }
        return redirect()->back()->with('success', $message);
    }

    /**
     * Apply bulk action on selected items
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function bulkAction(Request $request)
    {
        $content = new Content();

        if ($request->get('bulk-action') == 'active') {
            Content::whereIn('id', $request->get('id'))->update(['status' => 1]);
            $message = __('content.active_success');
        } elseif ($request->get('bulk-action') == 'inactive') {
            Content::whereIn('id', $request->get('id'))->update(['status' => 0]);
            $message = __('content.inactive_success');
        } elseif ($request->get('bulk-action') == 'delete') {
            Content::destroy($request->get('id'));
            $message = __('content.delete_success');
        }
        $content::flushCache($content);
        event(new BulkAction($content->getTable(), $request->get('id'), $request->get('bulk-action')));
        return redirect()->back()->with('success', $message);
    }

    public function getContentPageData($content)
    {
        $data['content'] = $content;
        $data['states'] = State::all();
        $data['approvedstates'] = Bar::where('status', 1)->get();
        $data['getStatesMinLength'] = collect($data['approvedstates'])->pluck('shortest_length_accepted', 'id')->toArray();
        return $data;
    }

    public function addNewSpeaker(Request $request) {
        try {
            $data = $request->validate([
                'name' => [
                    'required',
                    Rule::unique('faculty', 'name')->where(function ($query) use ($request) {
                        return $query->whereRaw('LOWER(name) = ?', [strtolower(trim($request->name))]);
                    }),
                ],
                'email' => 'required|email|regex:/^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$/'
            ], [
                'name.required' => 'The name field is required.',
                'name.unique' => 'This speaker name already exists.',
            ]);
    
            // Create new speaker
            $speaker = Faculty::create(['name' => trim($request->name), 'email' => trim($request->email)]);
    
            return response()->json([
                'success' => true,
                'id' => $speaker->id,
                'name' => $speaker->name,
            ], 201);
    
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'errors' => $e->errors(),
            ], 422);
        }
    }
}
