<?php

namespace App\Http\Controllers\Admin;

use App\Events\BulkAction;
use App\Http\Controllers\Controller;
use App\QREvents;
use App\QRStatistics;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Facades\Storage;
use SimpleSoftwareIO\QrCode\Facades\QrCode;

class EventStatisticsController extends Controller
{
    protected $indexRoute = 'events.index';
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $events = new QREvents();
        $events = $events->getResult($request);

        // Render view
        return view('admin.modules.eventstatistics.index')->with('events', $events);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function show(QREvents $event)
    {
        $statistics = $event->qrStatistics()->latest()->get();
        return view('admin.modules.eventstatistics.show')->with(['event'=> $event, 'statistics'=>$statistics]);
    }

    /*
     * Apply bulk action on selected items
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function bulkaction(Request $request)
    {
        $event = new QREvents();

        if ($request->get('bulk-action') == 'active') {
            QREvents::whereIn('id', $request->get('id'))->update(['status' => 1]);
            $message = __('event.active_success');
        } elseif ($request->get('bulk-action') == 'inactive') {
            $message = __('event.inactive_success');
            QREvents::whereIn('id', $request->get('id'))->update(['status' => 0]);
        } elseif ($request->get('bulk-action') == 'delete') {
            $needVerify = false;
            foreach ($request->get('id') as $pageId) {
                $event = QREvents::findOrFail($pageId);
                QREvents::destroy($pageId);
            }
            if (!$needVerify) {
                $message = __('event.delete_success');
            } else {
                $message = __('event.delete_success_validate');
            }
        }
        // $event::flushCache($event);
        event(new BulkAction($event->getTable(), $request->get('id'), $request->get('bulk-action')));
        return redirect()->back()->with('success', $message);
    }
    /**
     * Apply change status
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function changestatus(Request $request)
    {
        $event = QREvents::findOrFail($request->id);
        $event->status = $event->status == 1?0:1;
        $event->save();
        $message = displayMessages('event', $event->status);
        return redirect()->back()->with('success', $message);
    }
    
    public function downloadQr($id)
    {
        $event = QREvents::findOrFail($id);

        if (empty($event->qr_endpoint)) {
            return response()->json(['error' => 'QR endpoint is missing.'], 400);
        }

        $qrContent = $event->qr_endpoint;
        $folder = 'qrcodes';
        $filename = 'qrcode_' . $event->uuid . '.svg';
        $relativePath = $folder . '/' . $filename;
        $storagePath = storage_path('app/' . $relativePath);

        if (!File::exists(storage_path('app/' . $folder))) {
            File::makeDirectory(storage_path('app/' . $folder), 0755, true);
        }
        $qrSvg = QrCode::format('svg')->size(300)->generate($qrContent);

        Storage::disk()->put($relativePath, $qrSvg);
        $url = Storage::url($relativePath);

        $response = Response::make($qrSvg, 200, [
            'Content-Type' => 'image/svg+xml',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ]);

        Log::info('QR code saved at: ' . $storagePath);
        Log::info('QR code URL: ' . $url);
        
        return $response;
    }

}
