<?php

namespace App\Http\Controllers\Api;

use Validator;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Auth;
use App\Http\Controllers\Controller;
use App\AppUserAuthorization;
use App\Category;
use Laravel\Passport\Passport;
use GuzzleHttp\Exception\ClientException;
use App\Page;
use App\PracticeArea;
use App\State;

class BaseController extends Controller
{
    /**
     * Check APP Version
     *
     * @param Request $request
     * @return JSON
     */
    public function checkAppVersion(Request $request)
    {
        $this->validate($request, [
            'device_type' => ['required', Rule::in(['ios', 'android','web'])],
            'app_version' => ['required', function ($attribute, $value, $fail) use ($request) {
                if ($request->get('device_type') == 'ios') {
                    if (!version_compare(siteconfig('ios_app_version'), $value, '<=')) {
                        $fail(trans('New '.str_replace("_"," ",$attribute).' is available. Please update to the latest version of the app for a better experience.'));
                    }
                } elseif ($request->get('device_type') == 'android') {
                    if (!version_compare(siteconfig('android_app_version'), $value, '<=')) {
                        $fail(trans('New '.str_replace("_"," ",$attribute).' is available. Please update to the latest version of the app for a better experience.'));
                    }
                } elseif ($request->get('device_type') == 'wev') {
                    if (!version_compare(siteconfig('web_version'), $value, '<=')) {
                        $fail(trans('New '.str_replace("_"," ",$attribute).' is available. Please update to the latest version of the app for a better experience.'));
                    }
                }
            }]
        ]);
        return response()->json(['message' => trans("The app version is up to date.")]);
    }

    /**
     * Update Device Information
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response JSON
     */
    public function updateDeviceInfo(Request $request)
    {
        $this->validate($request, ['device_id' => 'required']);

        $data = $request->all();
        $existingRecords = AppUserAuthorization::where('user_id', $request->user()->token()->user_id)->get();

        if ($existingRecords->isNotEmpty()) {
            // Update all records with the latest device_token
            AppUserAuthorization::where('user_id', $request->user()->token()->user_id)->update([
                'device_token' => NULL,
            ]);
        }

        $data['oauth_access_token_id'] = $request->user()->token()->id;
        $data['user_id'] = $request->user()->token()->user_id;

        $app_user_authorization = AppUserAuthorization::where('oauth_access_token_id', $request->user()->token()->id)->first();
        $app_user_authorization = !empty($app_user_authorization) ? $app_user_authorization : new AppUserAuthorization();
        $app_user_authorization->fill($data);
        $app_user_authorization->save();

        return response()->json(['message' => trans("Device info saved successfully.")]);
    }


    /**
     * Get new access token
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response JSON
     */
    public function getNewAccessToken(Request $request)
    {
        /**
         * @body array{refresh_token: string, text: string}[]
         */
        $client = Passport::client()->where('password_client', 1)->get();
        $client = $client->sortByDesc('updated_at')->first();

        $http = new \GuzzleHttp\Client();

        try {
            $tokenResponse = $http->post(env('APP_URL'). '/oauth/token', [
                'form_params' => [
                    'grant_type'    => 'refresh_token',
                    'client_id'     => $client->id,
                    'client_secret' => $client->secret,
                    'refresh_token' => $request->refresh_token,
                    'scope'         => '',
                ]
            ]);

            $data = json_decode((string)$tokenResponse->getBody(), true);
            $response['message'] = trans("New token generated successfully.");
            $response['access_token'] = $data['access_token'];
            $response['refresh_token'] = $data['refresh_token'];

            return response()->json($response);
        }
        catch (ClientException $e) {
            $response = $e->getResponse();
            $responseBodyAsString = $response->getBody()->getContents();
            return response()->json($responseBodyAsString);
        }
    }

     /**
     * Get settings data
     *
     *  @return \Illuminate\Http\Response JSON
     */
    public function settings(){
        $slug = ["privacy","terms-of-use"];
        $pages = Page::whereIn('slug',$slug)->get();
        $data = [];
        foreach($pages as $page){
            $data[$page->slug == 'terms-of-use' ? 'terms_of_use' : $page->slug] = env('APP_URL') . "/page/". $page->slug;
        }
        $data['states'] =  State::all();
        $data['practiceareas'] =  PracticeArea::all();
        $data['category'] =  Category::all();

        $cl_type_arr = [];
        foreach(config('cl_type') as $key=>$value){
            $cl_type_arr[] = [
                'id' => $key,
                'label' => $value
            ];
        }
        $data['cl_type'] = $cl_type_arr;

        $provider_type_array = [];
        foreach(config('provider_type') as $key=>$value){
            $provider_type_array[] = [
                'id' => $key,
                'label' => $value
            ];
        }
        $data['provider_type'] =  $provider_type_array;

        $file_type_array = [];
        foreach(config('file_type') as $key=>$value){
            $file_type_array[] = [
                'id' => $key,
                'label' => $value
            ];
        }
        $data['file_type'] =  $file_type_array;

        return response()->json(['message' => "Data loaded successfully.",'data'=> $data ]);
    }

    function getFirmApprovedStatus() {
        $statuses = config('firmapprovalstatus');
        return response()->json( ["message"=> "Firm approved status data fetched successfully.",'data' => $statuses]);
    }
}
