<?php

namespace App\Http\Controllers\Api\v1;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Content;
use App\User;
use App\State;
use App\Bar;
use App\ContentListener;
use Illuminate\Validation\Rule;
use App\UserCredits;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;

use function PHPUnit\Framework\isNull;

class ContentController extends Controller
{
    protected $indexRoute = 'content.index';

    public function getActiveContent(Request $request)
    {
        $userBarIds = Auth::user()->getUserBarIds()->toArray();
    
        if (empty($userBarIds)) {
            return response()->json(['message' => "Currently user not registered in any bar."]);
        }
    
        $barIds = array_map('strval', $userBarIds);
        if ($request->state_id) {
            $barIds = array_intersect($barIds, array_map('strval', (array) $request->state_id));
        }
    
        $query = Content::query()
            ->where('status', 1)
            ->where(function ($query) {
                $query->whereNull('expire_date')
                      ->orWhereDate('expire_date', '>=', now()->toDateString());
            })
            ->where(function ($query) {
                $query->whereNull('temp_expire_date')
                      ->orWhere('temp_expire_date', '>=', now()->toDateString());
            })
            ->whereHas('contentsStates', function ($query) use ($barIds) {
                $query->whereIn('bar_id', $barIds);
            })
            ->when($request->id, fn ($q) => $q->where('id', $request->id))
            ->when($request->category_id, fn ($q) => $q->whereIn('category_id', (array) $request->category_id))
            ->when($request->cl_type, fn ($q) => $q->whereIn('cl_type', (array) $request->cl_type))
            ->with([
                'createdByUser:id,first_name,last_name',
                'contentFiles',
                'contentsStates:id,contents_id,bar_id',
                'category:id,name',
                'speakers.speaker' => function ($q) {
                    $q->where('status', 1)->get();
                },
            ])
            ->orderBy('published_on', 'desc');
    
        // If specific content requested
        if ($request->id) {
            $content = $query->first();
            if (!$content) {
                return response()->json(['message' => 'Content not found.'], 404);
            }
    
            $data = $this->formatContent($content);
            return response()->json(['message' => 'Content fetched successfully.', 'data' => $data]);
        }
    
        // Paginated result (temp_expire_date filtered in SQL)
        $perPage = 20;
        $contents = $query->get();
        $data = $contents->map(fn($content) => $this->formatContent($content));
    
        return response()->json([
            'message' => 'Contents fetched successfully.',
            'data' => $data,
            // 'current_page' => $contents->currentPage(),
            // 'last_page' => $contents->lastPage(),
            // 'per_page' => $contents->perPage(),
            // 'total' => $contents->total(),
        ]);
    }
    
    
    

    private function formatContent($content)
    {
        return [
            'id' => $content->id,
            'category_id' => $content->category_id,
            'category_name' => $content->category->name ?? null,
            'podcast_title' => $content->podcast_title,
            'cl_type' => $content->cl_type,
            'created_at' => $content->created_at,
            'podcast_status' => $content->podcast_status,
            'short_description' => $content->short_description,
            'internal_cl_podcast_id' => $content->internal_cl_podcast_id,
            'image_path' => $content->image_path,
            'image_alt' => $content->image_alt,
            'provider_name' => $content->provider_name,
            'content_files' => $content->contentFiles->map(function ($file) {
                return [
                    'id' => $file->id,
                    'contents_id' => $file->contents_id,
                    'length' => $file->length,
                    'credit' => $file->credit,
                    'credit_earned' => $file->credit_earned,
                    'view_counts' => $file->view_counts,
                    'file' => $file->file,
                    'file_name' => $file->file_name,
                    'file_path' => $file->file_path,
                    'last_listen_time_in_seconds' => $file->last_listen_time_in_seconds,
                    'created_at' => $file->created_at,
                    'status' => $file->status,
                ];
            }),
            'contents_states' => $content->contentsStates,
            'temp_expire_date' => $content->temp_expire_date,
            'speakers' => $content->speakers->filter(function ($s) {
                return $s->speaker && $s->speaker->status == 1;
            })->map(function ($s) {
                $speaker = $s->speaker;
                $speaker->description = strip_tags($speaker->description);
                return $speaker;
            })->values(),

        ];
    }
    

    public function getContentPageData($content)
    {
        $data['content'] = $content;
        $data['states'] = State::all();
        $data['approvedstates'] = Bar::where('status', 1)->get();
        $data['getStatesMinLength'] = collect($data['approvedstates'])->pluck('shortest_length_accepted', 'id')->toArray();
        return $data;
    }

    public function contentFileListen(Request $request)
    {

        $rules = [
            'content_id' => 'required',
            'content_file_id' => [
                'required',
                Rule::exists('content_upload', 'id')->where(function ($query) use ($request) {
                    if(!(env('APP_ENV') == 'local')){
                        $query->where('id', $request->input('content_file_id'))
                        ->where('length', '>=', $request->input('listen_time'));
                    }
                }),
            ],
            'listen_time' => 'required',
            'listen_status' => 'required',
        ];
        $messages = [
            'content_file_id.exists' => 'The listen time exceeds the length of the content file.',
        ];
        $this->validate($request, $rules, $messages);

        $contentListenerData = ContentListener::updateOrCreate(
            ['id' => $request->listener_id],
            [
                'content_id' => $request->content_id,
                'content_file_id' => $request->content_file_id,
                'user_id' => Auth::user()->id,
                'listen_time' => $request->listen_time, //secondsToMinuteSecond($request->listen_time)
                'listen_status' => $request->listen_status,
                'completed_at' => $request->listen_status == 1 ? Carbon::now() : NULL,
            ]
        );
        if ($contentListenerData) {
            $user_bar_ids = Auth::user()->getUserBarActiveIds()->toArray();
            if (count($user_bar_ids) == 0) {
                return response()->json(['message' => "Currently user not register or subscribe for this bar."]);
            }
            $array = array_map('strval', $user_bar_ids);
            if ($request->state_id) {
                $array = array_intersect($array, $request->state_id);
            }
            $allContentStateAssociatedWithUser = $contentListenerData->contents->contentsStates()->whereIn('bar_id', $array)->get();

            if (count($allContentStateAssociatedWithUser)) {
                $userID = Auth::user()->id;
                $creditsToInsert = [];
                foreach ($allContentStateAssociatedWithUser as $contentState) {
                    $creditInSeconds = $contentState->credit_provided_by_the_course * 60;

                    $userCredits = UserCredits::where('user_id', $userID)
                        ->where('content_id', $request->content_id)
                        ->where('content_file_id', $request->content_file_id)
                        ->where('content_state_id', $contentState->id)
                        ->orderBy('id', 'desc')
                        ->first();

                    $nextCreditAt = 0;
                    $lastCreditFor = 0;
                    if ($userCredits) {
                        $lastCreditFor = explode("-", $userCredits->credit_for)[1];
                        $nextCreditAt = $lastCreditFor + $creditInSeconds;
                    } else {
                        $nextCreditAt = $creditInSeconds;
                    }

                    $subscriptions = Auth::user()->subscriptions->where('bar_id',$contentState->bar_id)->where('stripe_subscription_status','active')->first();
                    $creditData = checkNeedToAddCredit($lastCreditFor, $nextCreditAt, $request->listen_time, $creditInSeconds);
                    $now = Carbon::now();
                    $loggedUser=Auth::user()->id ? Auth::user()->id : 0;
                    if (count($creditData)) {
                        foreach ($creditData as $value) {
                            $creditFor = explode("-", $value);
                            $creditsToInsert[] = [
                                'user_id' => $userID,
                                'content_id' => $request->content_id,
                                'content_file_id' => $request->content_file_id,
                                'content_state_id' => $contentState->id,
                                'bar_id' => $contentState->bar_id,
                                'subscription_id' => $subscriptions->id,
                                'stripe_subscription_id' => $subscriptions->subscription_id,
                                'content_listeners_id' => $contentListenerData->id,
                                'credit_for' => $value,
                                'credit' => getCreditUsingCreditFor($creditFor[0],$creditFor[1]),
                                'created_at'=> $now,
                                'updated_at'=> $now,
                                'created_by'=>$loggedUser,
                                'updated_by'=>$loggedUser,
                            ];
                        }
                    }
                }
                // Bulk insert user credits if needed
                if (!empty($creditsToInsert)) {
                    UserCredits::insert($creditsToInsert);
                }
            }
        }
        return response()->json(["message" => "Contents listen added successfully.", 'data' => $contentListenerData]);
    }

    // public function contentListenHistory(Request $request)
    // {
    //     $contentListener = ContentListener::where('user_id', Auth::user()->id)
    //         ->when($request->id, function ($query) use ($request) {
    //             return $query->where('content_id', $request->id);
    //         })->with('contents:id,podcast_title,provider_id,image,image_alt,expire_date', 'contentFiles:id,file,length')
    //         ->whereHas('contents', function ($query) use ($request) {
    //             $query->when($request->category_id, function ($query) use ($request) {
    //                 return $query->whereIn('category_id', $request->category_id);
    //             });
    //             $query->when($request->cl_type, function ($query) use ($request) {
    //                 return $query->whereIn('cl_type', $request->cl_type);
    //             });
    //             $query->when($request->state_id, function ($query) use ($request) {
    //                 return $query->whereHas('contentsStates', function ($query) use ($request) {
    //                     $query->whereIn('bar_id', $request->state_id);
    //                 });
    //             });
    //         })->orderByDesc('created_at')->get();

    //     $contentListener = $contentListener->filter(function ($listener) {
    //             return Carbon::parse($listener->contents->temp_expire_date)->gt(Carbon::now());
    //         })->values();

    //     return response()->json(["message" => "Contents listen history fetch successfully.", 'data' => $contentListener]);
    // }
    public function contentListenHistory(Request $request)
{
    $user_id = Auth::id();

    // Build base query for ContentListener
    $query = ContentListener::where('user_id', $user_id)
        ->with(['contents' => function ($query) {
            $query->select('id', 'podcast_title', 'provider_id', 'image', 'image_alt', 'expire_date');
        }, 'contentFiles' => function ($query) {
            $query->select('id', 'contents_id', 'file', 'length','view_file_name');
        }])
        ->orderByDesc('created_at');

    // Apply conditional filters using when() and whereHas()
    $query->when($request->id, function ($query) use ($request) {
        $query->where('content_id', $request->id);
    });

    $query->whereHas('contents', function ($query) use ($request) {
        $query->when($request->category_id, function ($query) use ($request) {
            $query->whereIn('category_id', $request->category_id);
        });
        $query->when($request->cl_type, function ($query) use ($request) {
            $query->whereIn('cl_type', $request->cl_type);
        });
        $query->when($request->state_id, function ($query) use ($request) {
            $query->whereHas('contentsStates', function ($query) use ($request) {
                $query->whereIn('bar_id', $request->state_id);
            });
        });
    });

    // Execute the query and retrieve results
    $contentListener = $query->get();

    // Filter and format the results
    $contentListener = $contentListener->filter(function ($listener) {
        if(!isNull($listener->contents->temp_expire_date)){
            return Carbon::parse($listener->contents->temp_expire_date)->gt(Carbon::now());
        } else {
            return true;
        }
    })->values();

    // Implement the filtering logic to remove duplicate podcast entries based on `content_file_id`
    $uniquePodcast = [];
    $contentListener = $contentListener->filter(function ($listener) use (&$uniquePodcast) {
        // Check if content_file_id already exists in the uniquePodcast array
        $foundSame = collect($uniquePodcast)->first(function ($item) use ($listener) {
            return $listener->content_file_id == $item['content_file_id'];
        });

        // If foundSame is null, it means the podcast is unique and should be added
        if (!$foundSame) {
            $uniquePodcast[] = [
                'content_id' => $listener->content_id,
                'content_file_id' => $listener->content_file_id,
                'updated_at' => $listener->updated_at,
                'content_files' => [
                    'last_listen_time_in_seconds' => $listener->contentFiles->last_listen_time_in_seconds ?? null,
                    'length' => $listener->contentFiles->length ?? null,
                    'file_name' => $listener->contentFiles->file_name ?? null,
                ],
                'contents' => [
                    'image_path' => $listener->contents->image_path ?? null,
                    'podcast_title' => $listener->contents->podcast_title ?? null,
                ],
            ];
        }
    });

    // Return the unique podcast data
    return response()->json([
        "message" => "Contents listen history fetched successfully.",
        'data' => $uniquePodcast
    ]);
}
}