<?php

namespace App\Http\Controllers\Api\v1;

use App\User;
use App\Events\BulkAction;
use App\Http\Controllers\Controller;
use App\Notifications\UserVerification;
use App\Notifications\FrontUserVerification;
use Illuminate\Support\Facades\Password;
use Illuminate\Mail\Message;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Carbon\Carbon;
use App\Userdocument;
use App\License;
use App\Course;
use App\EnrolCourseReference;
use App\Notifications\FrontUserVerificationFromCms;
use PDF;
use App\Help;
use App\GroupMember;
use App\CourseApproval;
use App\Notifications\OnboardingFrontUserVerification;
use Auth;
use App\BarUserReference;
use App\Bar;
use App\ContentListener;
use App\Firm;
use App\Notifications\AcceptWithoutPayFirmUserStatus;
use App\Notifications\AccountDeactivation;
use App\Notifications\FirmUserVerification;
use App\Notifications\RejectFirmUserStatus;
use App\Provider;
use App\Rules\CaseSensitiveExists;
use App\Submissions;
use App\Subscription as AppSubscription;
use App\UserCredits;
use Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Session;
use Laravel\Passport\Passport;
use Validator;

use Laravel\Passport\ClientRepository;
use Laravel\Passport\TokenRepository;
use NunoMaduro\Collision\Adapters\Phpunit\Subscribers\Subscriber;
use Psy\Readline\Hoa\Console;
use Stripe\Subscription;

class UserController extends Controller
{
    protected $indexRoute = 'user.index';
    protected $studentList = 'student.list';
    protected $firmList = 'firmuser.list';
    /**
     * Display a listing of the resource.
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $users = new User;
        $request->not_in_user_type = [2, 3];
        $users = $users->getResult($request);
        $resetLink = route('user.index', ['search' => '', 'status' => '']);
        // Render view
        return view('admin.modules.user.index')->with('users', $users)->with('reset_link', $resetLink);
    }

    public function firmuser(Request $request)
    {
        $users = new User;
        $request->not_in_user_type = [1, 3];
        $users = $users->getResult($request);
        $resetLink = route('user.index', ['search' => '', 'status' => '']);
        // Render view
        return view('admin.modules.user.firm-index')->with('users', $users)->with('reset_link', $resetLink);
    }

    /**
     * Display a listing of the resource.
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function student(Request $request)
    {
        $users = new User;
        $userType = $users->getAssociateType();
        $request->merge(["type" => 3]);
        $users = $users->getResult($request);
        $resetLink = route($this->studentList, ['search' => '', 'status' => '']);
        $activeBars = Bar::where('status', 1)->pluck('state_name', 'id');

        $firmApprovedStatuses = config('firmapprovalstatus');
        // Render view
        return view('admin.modules.user.student-index')->with(['users' => $users, 'activebars' => $activeBars, 'reset_link' => $resetLink, 'user_type' => $userType, 'firmApprovedStatuses' => $firmApprovedStatuses]);
    }

    function firmApprovedStatus(Request $request)
    {
        $user_id = $request->user_id;
        $status_id = $request->status_id;
        $user = User::find($user_id);
        $firm = Firm::where('code', $user->firmcode)->first();
        if ($user->firm_approved_status == 3) {
            Session::flash('info', 'You cannot update its status as it is rejected');
            return response()->json(['status' => 2]);
        }
        if ($status_id == 3) {
            $data = User::where('id', $user_id)->update([
                'firmcode' => null,
                'firm_approved_status' => $status_id
            ]);
            $user->notify(new RejectFirmUserStatus($user, $firm));
        } else {
            $data = User::where('id', $user_id)->update(['firm_approved_status' => $status_id]);
            $user->notify(new AcceptWithoutPayFirmUserStatus($user, $firm));
        }
        if ($data) {
            Session::flash('success', 'Status updated successfully');
            return response()->json(['status' => 1]);
        } else {
            Session::flash('error', 'Something went wrong!');
            return response()->json(['status' => 0]);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $user = new User();
        $userType = $user->getAssociateAdminRole();
        $loginfirmcode = '';
        if (Auth::user()->user_type_id == 2) {
            $loginfirmcode = Auth::user()->firm->code;
        }
        return view('admin.modules.user.addedit')->with('user', $user)->with('user_type', $userType)->with('loginfirmcode', $loginfirmcode);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\User $user
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request = $this->stripHtmlTags($request, User::$notStripTags);
        $this->validation($request);
        $user = User::where('email', $request->get('email'))->first();
        $data = $request->all();
        if (!isset($data['user_type_id'])) {
            $data['user_type_id'] = 1;
        }
        if (!empty($user->id)) {
            // Restore user account if already exist
            $user->restore();

            // Update user data.

            $data['deleted_at'] = null;
            $data['password'] = null;
            $data['email_verified_at'] = null;
            $data['status'] = 0;
            if (isset($data['birthdate']) && !empty($data['birthdate'])) {
                $data['birthdate'] = Carbon::parse($data['birthdate'])->format('Y-m-d');
            }
            $user->fill($data);
            $user->save();
            $message = 'user.restored_success';
        } else {
            // Save the User Data
            $user = new User;
            if ($data['user_type_id'] == 3) {
                $data['password'] = bcrypt(config('app.db_dft_auth'));
            }

            if (isset($data['birthdate']) && !empty($data['birthdate'])) {
                $data['birthdate'] = Carbon::parse($data['birthdate'])->format('Y-m-d');
            }
            $user->fill($data);
            $user->save();

            if ($user->user_type_id == 3) {
                $firmData = [
                    'name' => "",
                    'address1' => Null,
                    'address2' => Null,
                    'city' => Null,
                    'state_id' => 0,
                    'zipcode' => Null,
                ];
                Firm::updateOrCreate(
                    ['user_id' => $user->id],
                    $firmData
                );
                $message = 'user.student_create_success';
            } else {
                $message = 'user.create_success';
            }
        }

        // Generate Reset Password Token
        $token = app(\Illuminate\Auth\Passwords\PasswordBroker::class)->createToken($user);
        $user->remember_token = $token;
        $user->save();
        // Send Verification Email to User.
        if (in_array($user->user_type_id, [3, 4, 5, 6])) {
            // Send Verification Email to User.
            if ($user->user_type_id == 3) {
                $user->notify(new FrontUserVerification($token));
            } else {
                $user->notify(new UserVerification($token));
            }
        } else {
            // Send Verification Email to User.
            $user->notify(new UserVerification($token));
        }
        if ($this->apiUrl() !== false) {
            return response()->json(['success' => __($message, ['first_name' => $request->get('first_name'), 'last_name' => $request->get('last_name')])]);
        } else {
            switch ($user->user_type_id) {
                case 1:
                    $route = 'user.edit';
                    $routelist = $this->indexRoute;
                    break;
                case 2:
                    $route = 'firmuser.edit';
                    $routelist = $this->firmList;
                    break;
                case 3:
                    $route = 'student.edit';
                    $routelist = $this->studentList;
                    break;
            }
            if ($request->get('btnsave') == 'savecontinue') {

                return redirect()->route($route, ['user' => $user->id])->with("success", __($message, ['first_name' => $request->get('first_name'), 'last_name' => $request->get('last_name')]));
            } elseif ($request->get('btnsave') == 'save') {

                return redirect()->route($routelist)->with("success", __($message, ['first_name' => $request->get('first_name'), 'last_name' => $request->get('last_name')]));
            } else {
                return redirect()->route($routelist);
            }
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\User $user
     * @return \Illuminate\Http\Response
     */
    public function edit($userID)
    {
        $user = User::findOrFail($userID);
        $userType = $user->getAssociateAdminRole();
        $loginfirmcode = '';
        if (Auth::user()->user_type_id == 2) {
            $loginfirmcode = Auth::user()->firm->code;
        }
        return view('admin.modules.user.addedit')->with('user', $user)->with('user_type', $userType)->with('loginfirmcode', $loginfirmcode);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\User $user
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, User $user)
    {

        $request = $this->stripHtmlTags($request, User::$notStripTags);
        $id = $user->id;
        $this->validation($request, $id);

        $data = $request->all();
        $validate = 'yes';
        if ($user->email_verified_at == null) {
            if (isset($data['status']) && !empty($data['status']) &&  $data['status'] == 1) {
                $validate = 'not';
            }
            unset($data['status']);
        }
        switch ($user->user_type_id) {
            case 2:
                $msgterm = 'firmuser_';
                $route = 'firmuser.edit';
                $routelist = $this->firmList;
                break;
            case 3:
                $msgterm = 'student_';
                $route = 'student.edit';
                $routelist = $this->studentList;
                break;
            default:
                $msgterm = '';
                $route = 'user.edit';
                $routelist = $this->indexRoute;
                break;
        }
        if ($validate == 'not') {
            \Session::flash('success', __('user.' . $msgterm . 'update_status_not_valid', ['first_name' => $request->get('first_name'), 'last_name' => $request->get('last_name')]));
        } else {
            \Session::flash('success', __('user.' . $msgterm . 'update_success', ['first_name' => $request->get('first_name'), 'last_name' => $request->get('last_name')]));
        }

        if ($user->email_verified_at !== null) {
            if ($data['password'] !== null) {
                $data['password'] = bcrypt($request->get('password'));
            } else {
                unset($data['password']);
            }
        }
        unset($data['email']);
        // Save the User Data
        $user->fill($data);
        $user->save();
        if ($this->apiUrl() !== false) {
            return response()->json(['success' => 'Account updated successfully.']);
        } else {
            if ($request->get('btnsave') == 'savecontinue') {
                return redirect()->back();
            } elseif ($request->get('btnsave') == 'save') {
                return redirect()->route($routelist);
            } else {
                return redirect()->route($this->indexRoute);
            }
        }
    }

    /**
     * Send reset password email to user
     *
     * @param  string  $email
     * @return \Illuminate\Http\Response
     */
    public function resetPassword($email, $usertypeid)
    {
        $credentials = ['email' => $email, 'user_type_id' => $usertypeid];

        $user = User::where('email', $email)->where('user_type_id', $usertypeid)->first();

        if ($user->email_verified_at == null) {
            // Generate Reset Password Token
            $token = app(\Illuminate\Auth\Passwords\PasswordBroker::class)->createToken($user);
            if (in_array($usertypeid, [3])) {
                $user->remember_token = $token;
                $user->save();
                // Send Verification Email to User.
                $user->notify(new UserVerification($token));
            } else {
                // Send Verification Email to User.
                $user->notify(new UserVerification($token));
            }
            return redirect()->back()->with('success', __('user.activationlink_success', ['email' => $email]));
        } else {
            $response = Password::broker('users')->sendResetLink(
                $credentials
            );
            switch ($response) {
                case Password::RESET_LINK_SENT:
                    return redirect()->back()->with('success', __('user.resetpasswordlink', ['email' => $email]));
                case Password::INVALID_USER:
                default:
                    return redirect()->back()->withErrors(['email' => trans($response)]);
            }
        }
    }

    /**
     * Apply bulk action on selected user
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function bulkaction(Request $request)
    {
        $user = new User();
        $urlarr = explode('/', url()->previous());
        $previousroute = end($urlarr);
        switch ($previousroute) {
            case 'student':
                $usertype = 'student_';
                break;
            case 'firmuser':
                $usertype = 'firmuser_';
                break;
            default:
                $usertype = '';
                break;
        }
        if ($request->get('bulk-action') == 'delete') {
            User::destroy($request->get('id'));
            $message = __('user.' . $usertype . 'delete_success');
        } elseif ($request->get('bulk-action') == 'active') {
            $needVerify = false;
            foreach ($request->get('id') as $userId) {
                $user = User::findOrFail($userId);
                if ($user->email_verified_at == null) {
                    $needVerify = true;
                } else {
                    User::where('id', $userId)->update(['status' => 1]);
                }
            }
            if ($needVerify) {
                $message = __('user.' . $usertype . 'active_success_validate');
            } else {
                $message = __('user.' . $usertype . 'active_success');
            }
        } elseif ($request->get('bulk-action') == 'inactive') {
            User::whereIn('id', $request->get('id'))->update(['status' => 0]);

            foreach ($request->get('id') as $userId) {

                $this->_removeAccessTokenData($userId);
            }
            $message = __('user.' . $usertype . 'inactive_success');
        }
        \Session::flash('success', $message);
        event(new BulkAction($user->getTable(), $request->get('id'), $request->get('bulk-action')));
        return redirect()->back();
    }

    /**
     * Apply change status
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function changestatus(Request $request)
    {
        $user = User::findOrFail($request->id);
        $urlarr = explode('/', url()->previous());
        $previousroute = end($urlarr);
        if ($user->status  == 1) {
            User::where('id', $request->id)->update(['status' => 0]);
            if ($previousroute == 'student') {
                \Session::flash('success', __('user.student_account_inactivated_success'));
            } else {
                \Session::flash('success', __('user.account_inactivated_success'));
            }

            $this->_removeAccessTokenData($request->id);
        } else {
            if ($user->email_verified_at == null) {
                if ($previousroute == 'student') {
                    \Session::flash('error', __('user.student_account_not_validated'));
                } else {
                    \Session::flash('error', __('user.account_not_validated'));
                }
            } else {
                User::where('id', $request->id)->update(['status' => 1]);
                if ($previousroute == 'student') {
                    \Session::flash('success', __('user.student_account_activated_success'));
                } else {
                    \Session::flash('success', __('user.account_activated_success'));
                }
            }
        }
        return redirect()->back();
    }

    /**
     * remove individual User
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $user = User::findOrFail($id);

        if ($user->enrolCourses->count() > 0) {
            $courseIds = $user->enrolCourses->pluck('course_id')->toArray();
            $enrollCourse = new EnrolCourseReference();
            foreach ($courseIds as $courseid) {
                $enrollCourse->unenrolCourse($courseid, $user->id);
            }
        }
        //license
        $licence = new License();
        $licence->where('user_id', $id)->delete();

        // userdocuments
        if ($user->userdocumentReference->count() > 0) {
            $user->userdocumentReference->delete();
        }

        // helps
        $help = new Help();
        $help->where('user_id', $id)->delete();

        // group_members
        $groupMember = new GroupMember();
        $groupMember->where('user_id', $id)->delete();

        User::destroy($id);

        $this->_removeAccessTokenData($id);

        $message = __('user.delete_success_individual', ['first_name' => $user->first_name, 'last_name' => $user->last_name]);
        \Session::flash('success', $message);
        return redirect()->back();
    }

    function deleteUserAccount(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'old_password' => ['required', function ($attribute, $value, $fail) {
                if (!Hash::check($value, Auth::User()->password)) {
                    $fail(__('changepassword.validation_invalid_current_password', ['attribute' => str_replace("-", " ", $attribute)]));
                }
            }]
        ]);
        if ($validator->fails()) {
            return $validator->validate();
        }
        $user = User::find(Auth::user()->id);
        if (!$user) {
            return response()->json(['message' => 'User not found']);
        }

        if($user->subscriptions()->exists()){
            $user->subscriptions()->each(function ($subscription) {
                cancelUserSubscription($subscription->subscription_id, 0, 'User Deleted');
            });
        }

        //Delete Firm record
        // $firm = new Firm();
        // $firm->where('user_id', $user->id)->delete();

        // //Delete submissions record
        // $submissions = new Submissions();
        // $submissions->where('user_id', $user->id)->delete();

        // //Delete barReference record
        // $bar_user_references = new BarUserReference();
        // $bar_user_references->where('user_id', $user->id)->delete();

        // $content_listener = new ContentListener();
        // $content_listener->where('user_id', $user->id)->delete();

        // $user_credits = new UserCredits();
        // $user_credits->where('user_id', $user->id)->delete();

        // $subscription = new AppSubscription();
        // $subscription->where('user_id', $user->id)->delete();

        $username = "dummyUser_".time();
        $user->email = $username."@user.com";
        $user->password = "";
        $user->first_name = $username;
        $user->last_name = $username;
        $user->profile_image = "";
        $user->phone = "";
        $user->birthdate = "";
        $user->display_name = "";
        //&& $user->delete()
        if ($user->save() ) {
            $user->notify(new AccountDeactivation());
            return response()->json(['message' => 'User account deleted successfully']);
        } else {
            return response()->json(['message' => 'Something went wrong!']);
        }
    }


    /**
     * Remove access token of app users
     *
     * @param integer $userId
     */
    private function _removeAccessTokenData($userId)
    {
        if (\Schema::hasTable('oauth_refresh_tokens')) {
            $user = User::findOrFail($userId);
            $user->tokens->each(function ($token) {
                $token->revoke();
            });
        }
    }

    /**
     * Validate the Form
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    private function validation($request, $id = 0)
    {
        //get and use of minimum charctor value and password strength from site config
        $passwordValidationRule = ['required'];
        $passwordPolicy = getPasswordPolicy($passwordValidationRule);

        $rules = [
            'first_name' => 'required',
            'last_name' => 'required',
        ];

        if ($id > 0 && $request['password'] !== null && $request->has('password')) {
            $rules['password'] = $passwordPolicy['passwordValidationRule'];
        }

        if ($id > 0) {
            $rules['email'] = "required|email:rfc,dns|unique:users,email,{$id},id,deleted_at,NULL";
        } else {
            $rules['email'] = ['required', 'email:rfc,dns', Rule::unique('users')->whereNull('deleted_at')];
        }

        if ($request['user_type_id'] == 3) {
            $rules['firmcode'] = ['nullable', new CaseSensitiveExists('firms', 'code')];

        }

        if($request['phone'] !== null && $request->has('phone')) {
            $rules['phone'] = 'min:10';
        }


        $messages = [
            'phone.required_unless' => 'The phone field is required if prefered contact method is Call or Text.',
            'confirm_password.same' => 'The password and confirm password must match.',
            'password.regex' => __('user.password_invalid_' . $passwordPolicy['passwordStrength'] . '_msg', array('min' => $passwordPolicy['passwordMinLength'])),
        ];

        $this->validate($request, $rules, $messages);
    }

    public function licenseDelete($id, $userid)
    {
        $licence = new License();
        $licence->where('id', $id)->delete();
        return redirect()->route('student.edit', ['id' => $userid])->with("success", __('user.license_delete'));
    }

    public function studentProgress($id, Request $request)
    {
        $data = $request->all();
        $student = User::findOrFail($id);
        $courseIds = $student->enrolCourses->pluck('course_id')->toArray();
        $courses = new Course;
        $courses = $courses->getProgressResult($id, $courseIds, $data);
        return view('admin.modules.user.student-progress')->with('student', $student)->with('courses', $courses);
    }

    public function courseDescriptionPopup()
    {
        $course_id = $_GET['course_id'];
        $crc = Course::where('id', $course_id)->where('status', 1)->first();
        return response()->view('admin.includes.coursedescription', [
            'course' => $crc
        ]);
    }


    public function courseCertificatePopup($id)
    {

        $course_id = $_GET['course_id'];
        // get course completed certificate
        $enrollCourse = new EnrolCourseReference();
        $certificatedData = $enrollCourse->getCompletedCertificates($course_id, $id);

        $crc = Course::where('id', $course_id)->first();
        return response()->view('admin.includes.coursecertificates', [
            'certificates' => $certificatedData,
            'title' => $crc->title,
            'courseid' => $course_id,
            'id' => $id
        ]);
    }


    public function generateCertificatePDF($lid, $caid, $courseid, $id)
    {

        $enrollCourse = new EnrolCourseReference();
        $data = $enrollCourse->generateCompletedCertificate($lid, $caid, $courseid, $id);

        $customPaper = array(0, 0, 455.00, 638.00);
        $pdf = PDF::loadView('pdf.certificate', $data)->setPaper($customPaper, 'landscape');
        return $pdf->stream('certificate.pdf');
    }

    public function unenroleUser($courseid, $id)
    {
        if ($courseid != 0) {
            $enrollCourse = new EnrolCourseReference();
            $enrollCourse->unenrolCourse($courseid, $id);
        }

        $user = User::findOrFail($id);
        $message = __('user.unenrol_user_admin', ['first_name' => $user->first_name, 'last_name' => $user->last_name]);
        \Session::flash('success', $message);
        return redirect()->back();
    }

    /*
     * Edit student course progress
     */

    public function studentProgressEdit($id, $courseid)
    {
        $userId = $id;
        $courseId = $courseid;

        $user = User::findOrFail($id);
        $course = EnrolCourseReference::where(['course_id' => $courseId, 'user_id' => $userId])->first();
        return view('admin.modules.user.addedit-student-progress')->with('user', $user)->with('course', $course);
    }


    /*
     * Update student course progress
     */

    public function studentProgressUpdate(Request $request)
    {

        $rules = [
            'status' => 'required|numeric',
            'score' => 'nullable|numeric',
            'time_spent' => 'nullable|numeric',
        ];

        if ($request->current_status == 3 || $request->current_status == 2) {
            $message = [
                'status.required' => 'You can not change the course status because this course is already completed.',
            ];
        } else {
            $message = [
                'status.numeric' => 'The status field is required .',
            ];
        }

        $this->validate($request, $rules, $message);
        $data = $request->all();
        $course = EnrolCourseReference::where(['course_id' => $data['course_id'], 'user_id' => $data['user_id']])->first();
        $status = $data['status'];

        if ($status == 0) {
            if ($data['completed_date'] !== null) {
                $course->completion_date = date('Y-m-d', strtotime($data['completed_date']));
            } else {
                $course->completion_date = null;
            }
        } elseif ($status == 1) {
            //
        } elseif ($status == 2 || $status == 3) {

            if ($data['completed_date'] !== null) {
                $course->completion_date = date('Y-m-d', strtotime($data['completed_date']));
            } else {
                $course->completion_date = Carbon::now()->format('Y-m-d');
            }
        }

        $course->status = $status;
        $course->timespent = $data['time_spent'];
        $course->save();

        Course::where('id', $data['course_id'])->update(['master_score' => $data['score']]);

        $message = 'Course progress updated successfully';

        if ($request->get('btnsave') == 'savecontinue') {
            return redirect()->back()->with("success", $message);
        } elseif ($request->get('btnsave') == 'save') {
            return redirect()->route('student.progress', $data['user_id'])->with("success", $message);
        } else {
            return redirect()->route('student.progress', $data['user_id']);
        }
    }

    //enroll new course for student from backend
    public function studentEnrollCourse($id)
    {
        $user = User::findOrFail($id);
        $currentDate = Carbon::now()->format('y-m-d');
        $enrolledCourses = $user->enrolCourses->pluck('course_id')->toArray();
        $courseIds = CourseApproval::whereNotIn('course_id', $enrolledCourses)
            ->where('start_date', '<=', $currentDate)->where('end_date', '>=', $currentDate)->pluck('course_id')->toArray();
        $courses = Course::whereIn('id', $courseIds)->where('status', 1)->get();
        return view('admin.modules.user.enroll-course')->with(['user' => $user, 'courses' => $courses]);
    }

    //update student enrolled course
    public function studentEnrollCourseUpdate(Request $request, $id)
    {

        $rules = [
            'course_id' => 'required|array',
        ];
        $message = [
            'course_id.required' => 'The course field is required .',
        ];
        $this->validate($request, $rules, $message);
        $data = $request->all();

        foreach ($data['course_id'] as $courseId) {
            $enroll = EnrolCourseReference::where(['course_id' => $courseId, 'user_id' => $id])->first();
            $course = Course::findOrFail($courseId);
            if ($enroll == null) {
                $data['enrol_amount'] = $course->price;
                $data['user_id'] =  $id;
                $data['course_id'] = $courseId;
                $data['status'] = 0;
                $enrol = new EnrolCourseReference;
                $enrol->fill($data);
                $enrol->save();
            }
        }

        $message = 'Course enrolled successfully';

        if ($request->get('btnsave') == 'savecontinue') {
            return redirect()->back()->with("success", $message);
        } elseif ($request->get('btnsave') == 'save') {
            return redirect()->route('student.progress', $id)->with("success", $message);
        } else {
            return redirect()->route('student.progress', $id);
        }
    }


    public function mail_verifaction($token, $email)
    {
        if (Auth::guard('frontuser')->user()) {
            return redirect()->route('home');
        }
        $user = User::where('email', $email)->first();
        $response = array();
        if (!isset($user->id)) {
            $response['error'] = __('auth.failed');
            if (!Session::has('authredirection')) {
                Session::forget('authredirection');
                Session::put('authredirection', 'auth.failed');
            }
            \Session::flash('error', $response['error']);
        } else if ($user->remember_token != $token) {
            $response['error'] = __('auth.token_expired');
            if (!Session::has('authredirection')) {
                Session::forget('authredirection');
                Session::put('authredirection', 'auth.token_expired');
            }
            \Session::flash('error', $response['error']);
        } else {
            $user->remember_token = NULL;
            $user->email_verified_at = date("Y-m-d h:i:s");
            $user->status = 1;
            $user->save();
            if (!Session::has('authredirection')) {
                Session::forget('authredirection');
                Session::put('authredirection', 'auth.success_mail_verification');
            }
            $user->notify(new OnboardingFrontUserVerification($user));
            $response['success'] = __('auth.success_mail_verification');
            Session::flash('message', $response['success']);

            $token = $user->createToken('auth_token')->accessToken;
            if (isset($token)) {
                $token_url = env('CL_APP_URL') . '/accountactivated?access_token=' . $token;
            } else {
                $token_url = env('CL_APP_URL') . '/accountactivated';
            }
            Session::put('access_token_url', $token_url);
        }
        return redirect()->route('accountActivated');
    }

    public function barReference(Request $request)
    {
        $rules = [
            '*.bar_id' => 'required|integer|exists:bars,id',
            '*.bar_number' => ['required', Rule::unique('bar_user_references', 'bar_number')->where(function ($query) {
                if(request()->input('*.id') != 0){
                    return $query->where('bar_id', request()->input('*.bar_id'))->whereNot('id', request()->input('*.id'));
                }else{
                    return $query->where('bar_id', request()->input('*.bar_id'));
                }
            })]
        ];

        $messages = [
            '*.bar_id.required' => 'The bar field is required.',
            '*.bar_number.unique' => 'The '. __('submissions.bar_id').' has already been taken.',
        ];

        $this->validate($request, $rules, $messages);

        $datas = $request->all();
        $user_id = Auth::user()->id;
        foreach ($datas as $data) {
            $data['user_id'] = $user_id;
            BarUserReference::updateOrCreate(
                ['id' => $data['id']??0],
                $data
            );
        }
        $user = User::find($user_id);
        if (isset($user->firmcode)) {
            $firm = Firm::where('code', $user->firmcode)->first();
            $firm_admin = User::find($firm->user_id);
            $firm_admin->notify(new FirmUserVerification($user, $firm, $firm_admin));
        }
        return response()->json(['message' => 'BAR has been added to your profile successfully.']);
    }

    public function deleteBarFromUser(Request $request, $id)
    {
        $user_id = Auth::user()->id;
        // Find the model instance by ID
        $model = BarUserReference::where('bar_id', $id)->where('user_id', $user_id);
        // Delete the model
        $model->delete();
        // Return a success response
        return response()->json(['message' => 'Deleted successfully']);
    }


    public function getAdminUsers(Request $request)
    {

        $adminUsers = new User;
        $adminUsers = $adminUsers->getProviderAdminList();
        return response()->json(["message" => "Admin User fetched successfully.", 'data' => $adminUsers]);
    }

    public function getSponsoredList(Request $request)
    {
        $providersList = new Provider;
        $request->user_type = 1;
        $providersList = $providersList->getProvidersList($request);
        return response()->json(["message" => "Sponsor fetched successfully.", 'data' => $providersList]);
    }

    public function getPartnerList(Request $request)
    {
        $providersList = new Provider;
        $request->user_type = 2;
        $providersList = $providersList->getProvidersList($request);
        return response()->json(["message" => "Partner fetched successfully.", 'data' => $providersList]);
    }

    public function getSubscriptions()
    {
        $subscriptions = Auth::user()->subscriptions()->with('bar')->where('stripe_subscription_status','active')->where('status',1)->get();
        if ($subscriptions->count() > 0) {
            foreach ($subscriptions as $key => $subscription) {
                $subscription = retrieveStripeSubscription($subscription->subscription_id);
                $subscriptions[$key]['data'] = getNeededSubscriptionDataFromSubscription($subscription);
            }
        }
        return response()->json(["message" => "Subscription fetched successfully.", 'data' => $subscriptions]);
    }

    public function getBarSubscription($bar_id,$bar_user_references_id)
    {
        try{
            $subscription = AppSubscription::where('user_id',Auth::user()->id)->where('bar_id',$bar_id)->where('bar_user_references_id',$bar_user_references_id)->latest()->first();
            if($subscription){
                return response()->json(["status"=>true,"message" => "Subscription fetched successfully.", 'data' => $subscription]);
            }else{
                return response()->json(["status"=>false,"message" => "Not able to fetched subscription successfully."]);
            }

        } catch (\Exception $e) {
            return response()->json(['message' => $e->getMessage()]);
        }
    }
}
