<?php

namespace App\Http\Controllers\Api\v2;

use App\Content;
use App\Http\Controllers\Controller;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class DashboardController extends Controller
{    
    public function getAllRecords(Request $request)
    {
        $perPage = siteconfig('api_pagination');
        $currentPage = LengthAwarePaginator::resolveCurrentPage();
    
        $userBarIds = Auth::user()->getUserBarIds()->toArray();
        if (empty($userBarIds)) {
            return response()->json(['message' => "Currently user not registered in any bar."]);
        }
    
        $barIds = array_map('strval', $userBarIds);
        if ($request->state_id) {
            $barIds = array_intersect($barIds, array_map('strval', (array) $request->state_id));
        }
    
        $query = Content::query()
        ->where('status', 1)
        ->where(function ($q) {
            $q->whereNull('expire_date')
            ->orWhereDate('expire_date', '>=', now()->toDateString());
        })
        ->where(function ($q) {
            $q->whereNull('temp_expire_date')
            ->orWhere('temp_expire_date', '>=', now()->toDateString());
        })
        ->whereHas('contentsStates', function ($q) use ($barIds) {
            $q->whereIn('bar_id', $barIds);
        })
        ->when($request->category_id, fn ($q) => $q->whereIn('category_id', (array) $request->category_id))
        ->when($request->cl_type, fn ($q) => $q->whereIn('cl_type', (array) $request->cl_type))
        ->when($request->search, function (Builder $q, $search) {
            $q->where(function ($q2) use ($search) {
                $q2->where('podcast_title', 'like', '%' . $search . '%')
                ->orWhereHas('contentFiles', function ($q3) use ($search) {
                    $q3->where('view_file_name', 'like', '%' . $search . '%');
                });
            });
        })
        ->with([
            'contentsStates:id,contents_id,bar_id',
            'contentFiles' => function ($q) use ($request) {
                if ($request->search) {
                    $q->where('view_file_name', 'like', '%' . $request->search . '%');
                }
            }
        ])
        ->select([
            'id',
            'category_id',
            'podcast_title',
            'cl_type',
            'created_at',
            'image_alt',
            'provider_id',
            'provider_type',
        ])
        ->orderBy('published_on', 'desc')
        ->get();
    
        $search = $request->input('search', '');
        $podcasts = [];
        $episodes = [];
    
        foreach ($query as $content) {
            $podcasts[] = [
                'id' => $content->id,
                'podcast_title' => $content->podcast_title,
                'image_path' => $content->image_path,
                'image_alt' => $content->image_alt,
                'provider_name' => $content->provider_name,
                'created_at' => $content->created_at,
            ];
    
            foreach ($content->contentFiles as $file) {
                $fileSearchMatch = empty($search) || stripos($file->view_file_name, $search) !== false;
    
                if ($fileSearchMatch) {
                    $episodes[] = [
                        'id' => $file->id,
                        'contents_id' => $file->contents_id,
                        'length' => $file->length,
                        'credit' => $file->credit,
                        'content_name' => $content->podcast_title,
                        'file' => $file->file,
                        'file_name' => $file->file_name,
                        'file_path' => $file->file_path,
                        'last_listen_time_in_seconds' => $file->last_listen_time_in_seconds,
                        'provider_name' => $content->provider_name,
                        'image_path' => $content->image_path,
                        'content_states' => $content->contentsStates,
                        'created_at' => $file->created_at,
                    ];
                }
            }
        }
    
        $podcastCollection = collect($podcasts);
        $episodeCollection = collect($episodes);
    
        $maxCollection = $podcastCollection->count() >= $episodeCollection->count() ? $podcastCollection : $episodeCollection;
        $total = $maxCollection->count();
    
        $paginatedPodcasts = new LengthAwarePaginator(
            $podcastCollection->forPage($currentPage, $perPage)->values(),
            $total,
            $perPage,
            $currentPage
        );
    
        $paginatedEpisodes = new LengthAwarePaginator(
            $episodeCollection->forPage($currentPage, $perPage)->values(),
            $total,
            $perPage,
            $currentPage
        );
    
        return response()->json([
            'podcasts' => $paginatedPodcasts->items(),
            'episodes' => $paginatedEpisodes->items(),
            'current_page' => $paginatedPodcasts->currentPage(),
            'last_page' => $paginatedPodcasts->lastPage(),
            'per_page' => $paginatedPodcasts->perPage(),
            'total' => $paginatedPodcasts->total(),
        ]);
    }
    
        

    public function getFilteredContent(Request $request)
    {
        $userBarIds = Auth::user()->getUserBarIds()->toArray();
    
        if (empty($userBarIds)) {
            return response()->json(['message' => "Currently user not registered in any bar."]);
        }
    
        $barIds = array_map('strval', $userBarIds);
        if ($request->state_id) {
            $barIds = array_intersect($barIds, array_map('strval', (array) $request->state_id));
        }
    
        $query = Content::where('status', 1)
            ->where(function ($q) {
                $q->whereNull('expire_date')
                  ->orWhereDate('expire_date', '>=', now()->toDateString());
            })
            ->where(function ($q) {
                $q->whereNull('temp_expire_date')
                  ->orWhere('temp_expire_date', '>=', now()->toDateString());
            })
            ->whereHas('contentsStates', function ($q) use ($barIds) {
                $q->whereIn('bar_id', $barIds);
            })
            ->when($request->category_id, fn ($q) => $q->whereIn('category_id', (array) $request->category_id))
            ->when($request->cl_type, fn ($q) => $q->whereIn('cl_type', (array) $request->cl_type))
            ->with([
                'createdByUser:id,first_name,last_name',
                'contentFiles',
                'contentsStates:id,contents_id,bar_id',
                'category:id,name',
                'speakers.speaker:id,status,description',
            ])
            ->orderBy('published_on', 'desc');

            $results = $query->get();

            $filtered = $results->filter(function ($content) use ($request) {
                return is_null($request->podcast_status) || $content->podcast_status == $request->podcast_status;
            })->values();
            
            $currentPage = LengthAwarePaginator::resolveCurrentPage();
            $perPage = $perPage = siteconfig('api_pagination');
            
            $paginated = new LengthAwarePaginator(
                $filtered->forPage($currentPage, $perPage)->values(),
                $filtered->count(),
                $perPage,
                $currentPage,
                ['path' => url()->current()]
            );
            
            return response()->json([
                'data' => $paginated->map(fn ($content) => $this->formatContent($content)),
                'current_page' => $paginated->currentPage(),
                'last_page' => $paginated->lastPage(),
                'per_page' => $paginated->perPage(),
                'total' => $paginated->total(),
            ]);
            
    }
    private function formatContent($content)
    {
        return [
            'id' => $content->id,
            'category_id' => $content->category_id,
            'category_name' => $content->category->name ?? null,
            'podcast_title' => $content->podcast_title,
            'cl_type' => $content->cl_type,
            'created_at' => $content->created_at,
            'podcast_status' => $content->podcast_status,
            'short_description' => $content->short_description,
            'internal_cl_podcast_id' => $content->internal_cl_podcast_id,
            'image_path' => $content->image_path,
            'image_alt' => $content->image_alt,
            'provider_name' => $content->provider_name,
            'content_files' => $content->contentFiles->map(function ($file) {
                return [
                    'id' => $file->id,
                    'contents_id' => $file->contents_id,
                    'length' => $file->length,
                    'credit' => $file->credit,
                    'credit_earned' => $file->credit_earned,
                    'view_counts' => $file->view_counts,
                    'file' => $file->file,
                    'file_name' => $file->file_name,
                    'file_path' => $file->file_path,
                    'last_listen_time_in_seconds' => $file->last_listen_time_in_seconds,
                    'created_at' => $file->created_at,
                    'status' => $file->status,
                ];
            }),
            'contents_states' => $content->contentsStates,
            'temp_expire_date' => $content->temp_expire_date,
            'speakers' => $content->speakers->filter(function ($s) {
                return $s->speaker && $s->speaker->status == 1;
            })->map(function ($s) {
                $speaker = $s->speaker;
                $speaker->description = strip_tags($speaker->description);
                return $speaker;
            })->values(),

        ];
    }


}
