<?php

namespace App\Http\Controllers\Auth;

use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\User;
use Illuminate\Foundation\Auth\AuthenticatesUsers;
use Illuminate\Contracts\Cache\Repository as Cache;
use App\Notifications\FrontUserSecurityCode;
use Illuminate\Contracts\Session\Session;

class LoginController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Login Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles authenticating users for the application and
    | redirecting them to your home screen. The controller uses a trait
    | to conveniently provide its functionality to your applications.
    |
    */

    use AuthenticatesUsers;

    /**
     * Where to redirect users after login.
     *
     * @var string
     */
    protected $redirectTo = 'admin/dashboard';

    /**
     * Max invalid login attemps count
     *
     * @var int
     */
    protected $maxAttempts = 10;

    protected $decayMinutes = 1; // Default is 1

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct(Cache $cache)
    {
        //get maximum attempt value from site config
        $this->maxAttempts = !empty(siteconfig('login_attempt')) ? siteconfig('login_attempt') : 10;
        //get user block time value from site config
        $this->decayMinutes = !empty(siteconfig('user_block_time')) ? siteconfig('user_block_time') : 1;
        $this->middleware('guest')->except('logout');
        $this->cache = $cache;
    }

    /**
     * Change Login Form Template.
     *
     * Overwrite Illuminate\Foundation\Auth\AuthenticatesUsers Namespace's showLoginForm function.
     *
     * @return view
     *
     */
    public function showLoginForm()
    {
        return view('admin.modules.auth.login');
    }

    /**
     * Get the needed authorization credentials from the request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return array
     */
    protected function credentials(Request $request)
    {
        return [$this->username() => $request->{$this->username()}, 'password' => $request->password, 'status' => 1,'user_type_id'=>[1,2,4,5,6]];
    }


    /**
     * Get the needed authorization credentials from the request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function logout(Request $request)
    {
        Auth::user()->update(['securitycode' => '']);
        Auth::logout();
        \Session::flash('success', trans("Logout Successfully."));
        return redirect()->route('login');
    }

    /**
     * Handle a login request to the application.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Http\Response|\Illuminate\Http\JsonResponse
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function login(Request $request)
    {

        $this->validateLogin($request);

        // If the class is using the ThrottlesLogins trait, we can automatically throttle
        // the login attempts for this application. We'll key this by the username and
        // the IP address of the client making these requests into this application.
        if (method_exists($this, 'hasTooManyLoginAttempts') &&
            $this->hasTooManyLoginAttempts($request)) {
            $this->fireLockoutEvent($request);

            return $this->sendLockoutResponse($request);
        }

        if ($this->attemptLogin($request)) {
            if ($request->hasSession()) {
                $request->session()->put('auth.password_confirmed_at', time());
            }
            $user = $this->guard()->user();
            if(env('APP_DEBUG') == true || $user->id == 1){
                $user->update(['securitycode' => null]);
                return $this->sendLoginResponse($request);
            }
            else if($user->securitycode == '' || $user->securitycode == null){
                $securityCode = $this->random_string();
                $user->update(['securitycode' => $securityCode]);
                $user->notify(new FrontUserSecurityCode($securityCode));
                return redirect()->route('security-code.form');
            }
            else{
                return $this->sendLoginResponse($request);
            }
        }

        // If the login attempt was unsuccessful we will increment the number of attempts
        // to login and redirect the user back to the login form. Of course, when this
        // user surpasses their maximum number of attempts they will get locked out.
        $this->incrementLoginAttempts($request);

        return $this->sendFailedLoginResponse($request);
    }
    /**
      * Send the response after the user was authenticated.
      *
      * @param  \Illuminate\Http\Request  $request
      * @return \Illuminate\Http\Response
      */
    protected function sendLoginResponse(Request $request)
    {
        $request->session()->regenerate();

        $this->clearLoginAttempts($request);

        //remove reset Passowrd token
        $user = $this->guard()->user();

        app(\Illuminate\Auth\Passwords\PasswordBroker::class)->deleteToken($user);
        return $this->authenticated($request, $user)
                ?: redirect()->intended($this->redirectPath());

    }

    /**
     * generate random string
     */
    private function random_string()
    {
        return mt_rand(100000,999999);
    }
    /* Method override to send correct error messages
     * Get the failed login response instance.
     *
     * @param \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    protected function sendFailedLoginResponse(Request $request)
    {
        $user = User::where($this->username(), $request->{$this->username()})->first();

        if ($user && $user->status == 0) {
            return redirect()->back()
                ->withErrors([
                    'auth_failed' => __('Your account is inactive. Please activate your account to continue.')
                ]);
        }

        return redirect()->back()
            ->withErrors([
                'auth_failed' => __('auth.failed')
            ]);
    }

    /**
    * Redirect the user after determining they are locked out.
    *
    * @param  \Illuminate\Http\Request  $request
    * @return void
    *
    * @throws \Illuminate\Validation\ValidationException
    */
    protected function sendLockoutResponse(Request $request)
    {
        $strKey = $this->throttleKey($request); // Get current user key
        $intSeconds = $this->decayMinutes * 60; // Get block time and convert it to seconds
        $intAttempts = $this->limiter()->attempts($strKey); // Get current number of attempts
        if ($intAttempts == $this->maxAttempts) {
            $this->cache->forget($strKey . ':timer'); // Reset block time for seting new block time
            $this->limiter()->hit($strKey, $intSeconds); // Set new block time define in our system
        }

        $seconds = $this->limiter()->availableIn(
            $strKey
        );

        return redirect()->back()
            ->withErrors([
                'auth_throttle' => __('auth.throttle', ['seconds' => $seconds])
            ]);
    }

    function accountActivated() {
        return view('admin.modules.auth.account_activate');
    }
}
