<?php

namespace App\Http\Controllers\Auth;
use App\Firm;
use App\User;
use App\Http\Controllers\Controller;
use App\Notifications\FrontFirmUserVerification;
use Illuminate\Foundation\Auth\RegistersUsers;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Notifications\UserVerification;
use App\Notifications\FrontUserVerification;
use App\Rules\CaseSensitiveExists;
use Carbon\Carbon;
use Exception;

class RegisterController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Register Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles the registration of new users as well as their
    | validation and creation. By default this controller uses a trait to
    | provide this functionality without requiring any additional code.
    |
    */

    use RegistersUsers;

    /**
     * Where to redirect users after registration.
     *
     * @var string
     */
    protected $redirectTo = 'admin/dashboard';

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('guest');
    }

    /**
     * Show the application registration form.
     *
     * @return \Illuminate\Http\Response
     */
    public function showRegistrationForm()
    {
        return view('admin.modules.auth.register');
    }

    /**
     * Create a new user instance after a valid registration.
     *
     * @param  array  $data
     * @return \App\User
     */
    protected function create(Request $request)
    {
        try{
        $request = $this->stripHtmlTags($request, User::$notStripTags);
        $data = $request->all();

        $this->validation($request);
        $user = User::where('email', $request->get('email'))->first();

        if(empty($data['user_type_id'])){
            $data['user_type_id'] = 3;
        }
        // dd(!empty($user->id));
        if (!empty($user->id)) {
            // Restore user account if already exist
            $user->restore();

            // Update user data.
            $data['deleted_at'] = null;
            $data['password'] = null;
            $data['email_verified_at'] = null;
            $data['status'] = 0;
            if(isset($data['birthdate']) && !empty($data['birthdate'])){
                $data['birthdate'] = Carbon::parse($data['birthdate'])->format('Y-m-d');
            }
            $user->fill($data);
            $user->save();
            $message = 'user.restored_success';
        } else {
            // Save the User Data
            $user = new User;

            $data['password'] = bcrypt($data['password']);

            if(isset($data['birthdate']) && !empty($data['birthdate'])){
                $data['birthdate'] = Carbon::parse($data['birthdate'])->format('Y-m-d');
            }
            $user->fill($data);
            $user->save();

            if($user->user_type_id == 3){
                $firmData = [
                    'name' => "",
                    'address1' => Null,
                    'address2' => Null,
                    'city' => Null,
                    'state_id' => 0,
                    'zipcode' => Null,
                ];
                Firm::updateOrCreate(
                    ['user_id' => $user->id],
                    $firmData
                );
                $message = 'user.student_create_success';
            }
            else{
                $message = 'user.create_success';
            }
        }

        $token = app(\Illuminate\Auth\Passwords\PasswordBroker::class)->createToken($user);
        $user->remember_token = $token;
        $user->save();




        // Send Verification Email to User.
        if($user->user_type_id == 3){
            $firmData = [
                'name' => "",
                'address1' => Null,
                'address2' => Null,
                'city' => Null,
                'state_id' => 0,
                'zipcode' => Null,
            ];
            Firm::updateOrCreate(
                ['user_id' => $user->id],
                $firmData
            );
            if(isset($user->firmcode)){
                $firm = Firm::where('code', $user->firmcode)->first();

                $user->notify(new FrontFirmUserVerification($token, $firm->name));
            }
            else{
                $user->notify(new FrontUserVerification($token));
            }
        }
        else{
            $user->notify(new UserVerification($token));
        }

        return response()->json( ['message' =>'Your account has been created. To continue, please check your email and verify your account.']);
    }catch(Exception $e){
        return response()->json(['message' => $e->getMessage()], 500);
    }
    }

    private function validation($request)
    {
        //get and use of minimum charctor value and password strength from site config
        $passwordValidationRule = ['required'];
        $passwordPolicy = getPasswordPolicy($passwordValidationRule);

        $rules = [
            'first_name' => 'required',
            'last_name' => 'required',
            'email' => ['required', 'email:rfc,dns', Rule::unique('users')->whereNull('deleted_at')],
            'password' => $passwordPolicy['passwordValidationRule'],
            'confirm-password' => 'same:password',
            'firmcode' => ['nullable', new CaseSensitiveExists('firms', 'code')]
        ];

        $search = array("<span>", "</span>");
        $replace = array("", "  ");
        $newmsg = str_replace($search, $replace,trim( __('user.password_rule_'.$passwordPolicy['passwordStrength'].'_msg', array('min'=>$passwordPolicy['passwordMinLength']))));

        $messages = [
            'phone.required_unless' => 'The phone field is required if prefered contact method is Call or Text.',
            'confirm-password.same' => 'The password and confirm password must match.',
            'password.regex' => $newmsg
        ];

        $this->validate($request, $rules, $messages);
    }

    function resendActivation(Request $request){
        $request->validate([
            'email' => 'required',
        ]);

        $user = User::where('email', $request->get('email'))->first();
        if($user){
            if($user->status == 1){
                $message = 'User already verified. Please login to continue.';
                return response()->json( ['message' => $message]);
            }
            $token = app(\Illuminate\Auth\Passwords\PasswordBroker::class)->createToken($user);
            $user->remember_token = $token;
            $user->save();

            // Send Verification Email to User.
            if($user->user_type_id == 3){
                $user->notify(new FrontUserVerification($token));
            }
            else{
                $user->notify(new UserVerification($token));
            }
            $message = 'Activation mail resend successfully';
            return response()->json( ['message' => $message]);
        }
        else{
            return response()->json(['message' =>'user not found'], 402);
        }

    }

}
