<?php

namespace App\Http\Controllers\Common;

use App\Page;
use App\Events\BulkAction;
use App\Http\Controllers\Controller;
use App\Http\Middleware\SetSiteConfig;
use App\SiteConfig;
use App\Submissions;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\View;

class PageController extends Controller
{
    protected $indexRoute = 'page.index';
    /**
     * Display a listing of the resource.
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $pages = new Page;
        $pages = $pages->getResult($request);

        // Render view
        return view('admin.modules.page.index')->with('pages', $pages);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $page = new Page();
        return view('admin.modules.page.addedit')->with('page', $page);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request, Page $page)
    {
        $request = $this->stripHtmlTags($request, Page::$notStripTags);
        $rules = [
            'title' => 'required|max:100',
            'status' => 'required',
            'description' => 'required',
            'banner' => 'required_if:has_banner,1',
            'meta_title' => 'required|max:255',
            'meta_desc' => 'required',
        ];

        $messages = [
            'banner.required_if' => 'The banner field is required.',
            'description.required' => 'The page content field is required.',
            'meta_desc.required' => 'The meta description field is required.'
        ];

        $this->validate($request, $rules, $messages);

        $title = $request->get('title');
        // For unqiue slug- To create the auto slug
        $slug = getSlugText($title, $page);

        // Prepare Data Array
        $data = $request->all();
        $data['slug'] = $slug;
        $data['banner'] = !empty($request->get('has_banner')) ?  $request->get('banner') : null;
        $data['banner_alt'] = !empty($request->get('has_banner')) ?  $request->get('banner_alt') : null;

        // Save the Data
        $page->fill($data);
        $page->save();

        if ($request->get('btnsave') == 'savecontinue') {
            return redirect()->route('page.edit', ['page' => $page->id])->with("success", __('page.create_success', ['title'=>$request->get('title')]));
        } elseif ($request->get('btnsave') == 'save') {
            return redirect()->route($this->indexRoute)->with("success", __('page.create_success', ['title'=>$request->get('title')]));
        } else {
            return redirect()->route($this->indexRoute);
        }
    }


    /**
     * Display the specified resource.
     *
     * @param  string  $slug
     * @return \Illuminate\Http\Response
     */
    public function show($slug = 'home')
    {
        $content = Page::where('slug', $slug)->where('status', 1)->firstOrFail();
        return view('front.modules.page.index')->with('content', $content);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Page $page
     * @return \Illuminate\Http\Response
     */
    public function edit(Page $page)
    {
        return view('admin.modules.page.addedit')->with('page', $page);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Page $page
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Page $page)
    {
        $request = $this->stripHtmlTags($request, Page::$notStripTags);
        $id = $page->id;
        $rules = [
                    'title' => 'required|max:100',
                    'slug' =>  "required|unique:pages,slug,{$id},id,deleted_at,NULL",
                    'status' => 'required',
                    'description' => 'required',
                    'banner' => 'required_if:has_banner,1',
                    'meta_title' => 'required|max:255',
                    'meta_desc' => 'required',
                ];

        // Set Error Messages
        $messages = [
            'banner.required_if' => 'The banner field is required when has banner is filled.',
            'description.required' => 'The page content field is required.',
            'meta_desc.required' => 'The meta description field is required.'
        ];

        $this->validate($request, $rules, $messages);

        // Prepare Data Array
        $data = $request->all();
        $data['banner'] = !empty($request->get('has_banner')) ?  $request->get('banner') : null;
        $data['banner_alt'] = !empty($request->get('has_banner')) ?  $request->get('banner_alt') : null;
        unset($data['slug']);
        // Save the Data
        $page->fill($data);
        $page->save();

        if ($request->get('btnsave') == 'savecontinue') {
            return redirect()->back()->with("success", __('page.update_success', ['title'=>$request->get('title'), 'last_name'=>$request->get('last_name')]));
        } elseif ($request->get('btnsave') == 'save') {
            return redirect()->route($this->indexRoute)->with("success", __('page.update_success', ['title'=>$request->get('title')]));
        } else {
            return redirect()->route($this->indexRoute);
        }
    }


    /**
     * Apply bulk action on selected items
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function bulkaction(Request $request)
    {
        $page = new Page();

        if ($request->get('bulk-action') == 'active') {
            Page::whereIn('id', $request->get('id'))->update(['status' => 1]);
            $message = __('page.active_success');
        } elseif ($request->get('bulk-action') == 'inactive') {
            $message = __('page.inactive_success');
            Page::whereIn('id', $request->get('id'))->update(['status' => 0]);
        } elseif ($request->get('bulk-action') == 'delete') {
            $needVerify = false;
            foreach ($request->get('id') as $pageId) {
                $page = Page::findOrFail($pageId);
                if ($page->slug == 'home') {
                    $needVerify= true;
                } else {
                    Page::destroy($pageId);
                }
            }
            if (!$needVerify) {
                $message = __('page.delete_success');
            } else {
                $message = __('page.delete_success_validate');
            }
        }
        $page::flushCache($page);
        event(new BulkAction($page->getTable(), $request->get('id'), $request->get('bulk-action')));
        return redirect()->back()->with('success', $message);
    }
    /**
     * Apply change status
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function changestatus(Request $request)
    {
        $page = Page::findOrFail($request->id);
        $page->status = $page->status == 1?0:1;
        $page->save();
        $message = displayMessages('page', $page->status);
        return redirect()->back()->with('success', $message);
    }

    public function appRedirect()
    {
        $userAgent = $_SERVER['HTTP_USER_AGENT'];

        if (stripos($userAgent, 'android') !== false) {

            $androidRedirectLinkCount = SiteConfig::where('name', 'androidRedirectLinkCount')->first();
            if(isset($androidRedirectLinkCount)) {
                $androidRedirectLinkCount->val = $androidRedirectLinkCount->val + 1;
                $androidRedirectLinkCount->save();
            }
            // Redirect to Google Play Store
            header('Location: https://play.google.com/store/apps/details?id='.siteconfig('android_app_id'));
        } elseif (stripos($userAgent, 'iphone') !== false || stripos($userAgent, 'ipad') !== false || stripos($userAgent, 'ipod') !== false) {
            $iosRedirectLinkCount = SiteConfig::where('name', 'iosRedirectLinkCount')->first();
            if(isset($iosRedirectLinkCount)) {
                $iosRedirectLinkCount->val = $iosRedirectLinkCount->val + 1;
                $iosRedirectLinkCount->save();
            }
            // Redirect to Apple App Store
            header('Location: https://apps.apple.com/us/app/'.siteconfig('ios_app_id'));
        } else {
            $unsupportedRedirectLinkCount = SiteConfig::where('name', 'unsupportedRedirectLinkCount')->first();
            if(isset($unsupportedRedirectLinkCount)) {
                $unsupportedRedirectLinkCount->val = $unsupportedRedirectLinkCount->val + 1;
                $unsupportedRedirectLinkCount->save();
            }
            // Handle other cases or fallback
            header('Location: '.env('CL_APP_URL'));
        }
        exit;
    }


    public function downloadCertificate($id)
    {
        $submission = Submissions::find(base64_decode($id));
        if($submission){
            $submission->certification_url = "";
            return view('pdf.certificate',['submission'=>$submission]);
        }
    }
}
