<?php

namespace App\Http\Controllers\Common;

use App\Bar;
use App\BarUserReference;
use App\Firm;
use App\Http\Controllers\Controller;
use App\Notifications\AcceptPayFirmUserStatus;
use App\Notifications\SendUserActivationEmailToAdmin;
use App\Notifications\SendUserDeactivationEmailToAdmin;
use App\Notifications\WelcomeEmail;
use App\Subscription;
use App\User;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Stripe\Exception\CardException;

class StripeController extends Controller
{

    function createPaymentLinkFunction($user_id,$redirectRoute){

        $user = Auth::user();
        if(!$user->stripe_customer_id){
            $stripe_customer_name = $user->fullName;
            if($user->user_type_id == 2){
                $stripe_customer_name = $user->firm->name." ( Firm )";
            }
            $customerRes = createStripeCustomer([
                'name'=>$stripe_customer_name,
                'email'=>$user->email
            ]);
            User::where('id', Auth::user()->id)->update([
                'stripe_customer_id' => $customerRes->id
            ]);
            Auth::user()->stripe_customer_id = $customerRes->id;
            $user = Auth::user();
        }
        $priceList = retrievePriceListOfUser($user_id);
        $redirectUrl = route($redirectRoute, ['user_id'=>$user_id,'session_id' => '{CHECKOUT_SESSION_ID}']);

        $promocodeId = siteconfig('stripe_discount_key') ?: null;

        $link = createPaymentLink($priceList, $redirectUrl, Auth::user()->stripe_customer_id, $promocodeId);
        

        return $link->url;
    }
    function checkout($user_id)
    {
        try {
            $user = User::find($user_id);
            $bar_ids = $user->getUserBarIdsNeedToPay();
            if (count($bar_ids) <= 0) {
                return redirect()->route('student.list');
            }
            $redirectRoute = "firmPaySuccess";
            $link = $this->createPaymentLinkFunction($user_id,$redirectRoute);
            return redirect()->away($link);
        } catch (Exception $e) {
            throw new Exception('Unexpected error: ' . $e->getMessage());
        }
    }

    function checkoutMobile()
    {
        try {
            $user = User::find(Auth::user()->id);
            $bar_ids = $user->getUserBarIdsNeedToPay();
            if (count($bar_ids) <= 0) {
                return response()->json(['status' => false, 'message' => "Not able to get your Price list for subscription."]);
            }
            $redirectRoute = "mobilePaySuccess";

            $link = $this->createPaymentLinkFunction(Auth::user()->id,$redirectRoute);

            return response()->json(['status' => true, 'message' => '', 'data' => ['paymentLink'=>$link]]);
        } catch (Exception $e) {
            return response()->json(['status' => false, 'message' => $e->getMessage()]);
        }
    }

    //if firm is pay for user then redirection should be here.
    function firmPaySuccess(){
        $stripe = new \Stripe\StripeClient(env('STRIPE_SECRET_KEY'));
        $session = $stripe->checkout->sessions->retrieve($_GET['session_id']);

        $user = User::find($_GET['user_id']);
        if($session->status == 'complete'){
            $subscription = retrieveStripeSubscription($session->subscription);
            sendSubscriptionInvoice(Auth::user()->id, $subscription->latest_invoice);
            // Insert subscription details in db
            $this->userSubscriptionAdd($subscription, $user->id,'firm');

            //Update user firm approve status
            User::where('id', $user->id)->update([
                'firm_approved_status' => '1'
            ]);
            $firm = Firm::where('code', $user->firmcode)->first();
            $user->notify(new AcceptPayFirmUserStatus($user, $firm));
            $user->notify(new WelcomeEmail($user));

            //send activation email to admin 
            $emails = siteconfig('activation_email_admin');
            if (!empty($emails)) {
                $emailsArray = explode(',', $emails);
                foreach ($emailsArray as $email) {
                    $admin = User::make(['email' => $email]);
                    if ($admin) {
                        $admin->notify(new SendUserActivationEmailToAdmin($user, $subscription));
                    }else {
                        $this->warn("User with email $email not found.");
                    }
                }
            }

            return redirect()->route('student.list')->with("success", __('user.firm_user_payment_success'));
        }else{
            return redirect()->route('student.list')->with("error", __('user.firm_user_payment_fail'));
        }
    }

    //if user pay form mobile then redirection should be here.
    function mobilePaySuccess(){
        $stripe = new \Stripe\StripeClient(env('STRIPE_SECRET_KEY'));
        $session = $stripe->checkout->sessions->retrieve($_GET['session_id']);

        $user = User::find($_GET['user_id']);
        if($session->status == 'complete'){
            $subscription = retrieveStripeSubscription($session->subscription);
            sendSubscriptionInvoice($_GET['user_id'], $subscription->latest_invoice);
            // Insert subscription details in db
            $this->userSubscriptionAdd($subscription, $user->id,'user');
            $user->notify(new WelcomeEmail($user));

            //send activation email to admin 
            $emails = siteconfig('activation_email_admin');
            if (!empty($emails)) {
                $emailsArray = explode(',', $emails);
                foreach ($emailsArray as $email) {
                    $admin = User::make(['email' => $email]);
                    if ($admin) {
                        $admin->notify(new SendUserActivationEmailToAdmin($user, $subscription));
                    }
                }
            }

            return view('admin.payment.success');
        }else{
            return view('admin.payment.failed');
        }
    }


    function renewalSubscription($user_id,$subscription_id,$bar_user_references_id)
    {
        try {
            $priceList = retrievePriceListOfUser($user_id,$bar_user_references_id,1);
            $redirectUrl = route('firmPayRenewalSuccess', ['user_id'=>$user_id,'session_id' => '{CHECKOUT_SESSION_ID}','subscription_id'=>$subscription_id,'bar_user_references_id'=>$bar_user_references_id]);

            $link = createPaymentLink($priceList,$redirectUrl,Auth::user()->stripe_customer_id, null);
            return redirect()->away($link->url);
        } catch (Exception $e) {
            throw new Exception('Unexpected error: ' . $e->getMessage());
        }

    }

    function firmPayRenewalSuccess(){
        $stripe = new \Stripe\StripeClient(env('STRIPE_SECRET_KEY'));
        $session = $stripe->checkout->sessions->retrieve($_GET['session_id']);

        $user = User::find($_GET['user_id']);
        if($session->status == 'complete'){
            $subscription = retrieveStripeSubscription($session->subscription);
            sendSubscriptionInvoice(Auth::user()->id, $subscription->latest_invoice);


            //cancel old subscription
            $subscriptionToCancel = Subscription::find($_GET['subscription_id']);

            if ($subscriptionToCancel) {
                if($subscriptionToCancel->stripe_subscription_status == 'active'){
                    cancelUserSubscription($subscriptionToCancel->subscription_id,0,'Renew');
                }
            }

            // Insert subscription details in db
            $this->userSubscriptionAdd($subscription, $user->id,'firm',$_GET['bar_user_references_id'],1);

            //Update user firm approve status
            // User::where('id', $user->id)->update([
            //     'firm_approved_status' => '1'
            // ]);
            // $firm = Firm::where('code', $user->firmcode)->first();
            // $user->notify(new AcceptPayFirmUserStatus($user, $firm));
            session()->flash('reload', 1);
            return redirect()->route('subscriptions.index')->with("success", __('user.firm_user_renewal_payment_success'));
        }else{
            return redirect()->route('subscriptions.index')->with("error", __('user.firm_user_payment_fail'));
        }
    }

    function renewalSubscriptionMobile($subscription_id,$bar_user_references_id)
    {
        try {
            $user_id = Auth::user()->id;
            $priceList = retrievePriceListOfUser($user_id,$bar_user_references_id,1);
            $redirectUrl = route('mobilePayRenewalSuccess', ['user_id'=>$user_id,'session_id' => '{CHECKOUT_SESSION_ID}','subscription_id'=>$subscription_id,'bar_user_references_id'=>$bar_user_references_id]);

            $link = createPaymentLink($priceList,$redirectUrl,Auth::user()->stripe_customer_id, null);

            return response()->json(['status' => true, 'message' => '', 'data' => ['paymentLink'=>$link->url]]);
        } catch (Exception $e) {
            return response()->json(['status' => false, 'message' => $e->getMessage()]);
        }

    }

    function mobilePayRenewalSuccess(){


        $stripe = new \Stripe\StripeClient(env('STRIPE_SECRET_KEY'));
        $session = $stripe->checkout->sessions->retrieve($_GET['session_id']);

        $user = User::find($_GET['user_id']);
        if($session->status == 'complete'){
            $subscription = retrieveStripeSubscription($session->subscription);
            sendSubscriptionInvoice($user->id, $subscription->latest_invoice);


            //cancel old subscription
            $subscriptionToCancel = Subscription::find($_GET['subscription_id']);

            if ($subscriptionToCancel) {
                if($subscriptionToCancel->stripe_subscription_status == 'active'){
                    cancelUserSubscription($subscriptionToCancel->subscription_id,0,'Renew');
                }
            }

            $this->userSubscriptionAdd($subscription, $user->id,'user',$_GET['bar_user_references_id'],1);

            return view('admin.payment.success');
        }else{
            return view('admin.payment.failed');
        }
    }

    function createSubscription(Request $request)
    {

        $this->validation($request);
        $payment_method_id  = $request->payment_method_id;
        $request->request->add(['user_id' => Auth::user()->id]);
        $user = User::find($request->user_id);
        $priceList = retrievePriceListOfUser($request->user_id,$request->bar_user_references_id);

        if (!empty($priceList)) {


            $customerId = $user->stripe_customer_id;
            $request->request->add(['stripe_customer_id' => $user->stripe_customer_id]);


            // Create subscription in stripe
            try {
                // Create subscription in stripe
                $subscriptionResult = createStripeSubscription($customerId,$priceList,$request->payment_method_id);

                sendSubscriptionInvoice($request->user_id, $subscriptionResult->latest_invoice);

                if ($subscriptionResult['status'] == false) {
                    return response()->json(['status' => false, 'message' => $subscriptionResult['error']]);
                }
                // Insert subscription details in db
                $this->userSubscriptionAdd($subscriptionResult, $request->user_id,'user');

                return response()->json(['status' => true, 'message' => 'Your subscription has been created successfully,Please check your email for complete your payment to activate your subscription.']);
            } catch (Exception $e) {
                return response()->json(['status' => false, 'message' => $e->getMessage()]);
            }


        }else{
            return response()->json(['status' => true, 'message' => 'Invalid plan id']);
        }
    }

    //we can use this function to add User Subscription in our table
    function userSubscriptionAdd($subscriptiondata, $user_id,$from,$bar_user_references_id=0,$ForForcedRenew=0)
    {
        $subscriptionResult = $subscriptiondata;
        $startDate = date('Y-m-d H:i:s', $subscriptionResult->current_period_start);
        $endDate   = date('Y-m-d H:i:s', $subscriptionResult->current_period_end);
        $user = User::find($user_id);

        $planIds = $user->getUserBarIdsNeedToPay($bar_user_references_id,$ForForcedRenew);

        $paid_by = 1;
        if($from == 'firm'){
            $paid_by = 2;
        }

        foreach ($planIds as $plan_id) {
            $barUserReference = BarUserReference::where('user_id',$user_id)->where('bar_id', $plan_id)->first();
            if ($barUserReference) {

                $promocode ='';
                if(!empty($subscriptionResult->discount)){
                    $promocodeData = retrieveStripePromocode($subscriptionResult->discount->promotion_code);
                    $promocode = $promocodeData->code;
                }

                $SubscriptionArray = array(
                    'user_id'       => $user_id,
                    'bar_id'         => $plan_id,
                    'bar_user_references_id' => $barUserReference->id,
                    'hours_needed_per_cycle' => $barUserReference->bar->hours_needed_per_cycle,
                    'general'         => $barUserReference->bar->general,
                    'ethics'         => $barUserReference->bar->ethics,
                    'remaining_hours_needed_per_cycle' => $barUserReference->bar->hours_needed_per_cycle,
                    'remaining_general'         => $barUserReference->bar->general,
                    'remaining_ethics'         => $barUserReference->bar->ethics,
                    'latest_invoice'         => $subscriptionResult->latest_invoice,
                    'couponcode'         => !empty($subscriptionResult->discount) ? $subscriptionResult->discount->coupon->name : '',
                    'promocode'         =>$promocode,
                    'cycle_length'         => $barUserReference->bar->cycle_length,
                    'subscription_id' => $subscriptionResult->id,
                    'stripe_customer_id'  => $subscriptionResult->customer,
                    'start_date'      => $startDate,
                    'end_date'        => $endDate,
                    'stripe_subscription_status'=> $subscriptionResult->status,
                );
                Subscription::create($SubscriptionArray);
                $barUserReference->is_paid = 1;
                $barUserReference->paid_by = $paid_by;
                $barUserReference->save();
            }
        }
    }

    private function validation($request)
    {
        $rules = [
            'payment_method_id' => 'required',
        ];

        $this->validate($request, $rules);
    }

    private function cancelSubscriptionValidation($request)
    {
        $rules = [
            'subscription_id' => 'required',
            'action' => 'required',
        ];

        $this->validate($request, $rules);
    }

    public function cancelSubscription(Request $request)
    {
        // Validate the request data
        $this->cancelSubscriptionValidation($request);

        // Find the subscription by ID
        $subscription = Subscription::find($request->subscription_id);

        if ($subscription) {
            try {
                // Call the cancelUserSubscription function to handle cancellation
                cancelUserSubscription($subscription->subscription_id, $request->action);

                //send deactivation email to admin 
                $user = User::find(Auth::user()->id);
                $emails = siteconfig('activation_email_admin');
                if (!empty($emails)) {
                    $emailsArray = explode(',', $emails);
                    foreach ($emailsArray as $email) {
                        $admin = User::make(['email' => $email]);
                        if ($admin) {
                            $admin->notify(new SendUserDeactivationEmailToAdmin($user));
                        } else {
                            $admin->warn("User with email $email not found.");
                        }
                    }
                }

                // Return success response
                return response()->json(['status' => true, 'message' => 'Your subscription has been updated successfully']);
            } catch (Exception $e) {
                throw new Exception('Unexpected error: ' . $e->getMessage());
            }
        } else {
            // Subscription not found, return error response
            return response()->json(['status' => false, 'message' => 'Subscription not found'], 404);
        }
    }

    private function reNewValidation($request)
    {
        $rules = [
            'bar_user_references_id' => 'required',
            'subscription_id' => 'required',
            'payment_method_id' => 'required',
        ];

        $this->validate($request, $rules);
    }

    function reNewSubscription(Request $request)
    {

        $this->reNewValidation($request);
        $request->request->add(['user_id' => Auth::user()->id]);
        $user = User::find($request->user_id);
        $priceList = retrievePriceListOfUser($request->user_id,$request->bar_user_references_id,1);

        if (!empty($priceList)) {

            $customerId = $user->stripe_customer_id;
            $request->request->add(['stripe_customer_id' => $user->stripe_customer_id]);

            // Create subscription in stripe
            try {
                // Call the function to create the Stripe subscription
                $subscriptionResult = createStripeSubscription($customerId, $priceList,$request->payment_method_id);

                if ($subscriptionResult['status'] == false) {
                    return response()->json(['status' => false, 'message' => $subscriptionResult['error']]);
                }

                sendSubscriptionInvoice($request->user_id, $subscriptionResult->latest_invoice);

                //cancel old subscription
                $subscription = Subscription::find($request->subscription_id);

                if ($subscription) {
                    if($subscription->stripe_subscription_status == 'active'){
                        cancelUserSubscription($subscription->subscription_id,0,'Renew');
                    }
                }

                // Insert subscription details in db
                $this->userSubscriptionAdd($subscriptionResult, $request->user_id,'user',$request->bar_user_references_id,1);


                return response()->json(['status' => true, 'message' => 'Your subscription has been renewed successfully.']);
            } catch (Exception $e) {
                return response()->json(['status' => false, 'message' => $e->getMessage()]);
            }

        }else{
            return response()->json(['status' => true, 'message' => 'Invalid plan id']);
        }
    }

    function getPaymentMethodsOfUser(Request $request){
        try {
            $user = Auth::user();
            if(!$user->stripe_customer_id){
                return response()->json(["message" => "No Payment Methods Found.", 'data' => []]);
            }
            $cards = getAllPaymentMethodOfUser($user->stripe_customer_id);
            return response()->json(["message" => "Payment Methods fetched successfully.", 'data' => $cards]);
        } catch (Exception $e) {
            return response()->json(['status' => false, 'message' => $e->getMessage()]);
        }
    }

    function addNewCardToUser(Request $request){
        try {
            $user = Auth::user();
            if(!$user->stripe_customer_id){
                $stripe_customer_name = $user->fullName;
                if($user->user_type_id == 2){
                    $stripe_customer_name = $user->firm->name." ( Firm )";
                }
                $customerRes = createStripeCustomer([
                    'name'=>$stripe_customer_name,
                    'email'=>$user->email,
                    //'source' => $request->stripe_token
                ]);
                User::where('id', Auth::user()->id)->update([
                    'stripe_customer_id' => $customerRes->id
                ]);
                $customerId = $customerRes->id;
                Auth::user()->stripe_customer_id = $customerRes->id;
                $user = Auth::user();
            }else{
                $customerId = $user->stripe_customer_id;
                $request->request->add(['stripe_customer_id' => $user->stripe_customer_id]);
            }
            $card = addNewCardToUser($customerId,$request->stripe_token);
            return response()->json(["message" => "Payment Methods added successfully.", 'data' => $card]);
        } catch (Exception $e) {
            return response()->json(['status' => false, 'message' => $e->getMessage()]);
        }
    }

    function deleteCardFromUser(Request $request){
        try {
            $stripe = new \Stripe\StripeClient(env('STRIPE_SECRET_KEY'));
            $stripe->paymentMethods->detach($request->payment_method_id, []);
            return response()->json(["message" => "Payment Methods deleted successfully.", 'data' => []]);
        } catch (Exception $e) {
            return response()->json(['status' => false, 'message' => $e->getMessage()]);
        }
    }

    function cancelSubscriptionByFirm($subscription_id){

        $subscription = Subscription::find($subscription_id);

        if ($subscription) {
            try {
                cancelUserSubscription($subscription->subscription_id, 1);

                $userid = $subscription->user_id;

                //send deactivation email to admin 
                $user = User::find($userid);
                $emails = siteconfig('activation_email_admin');
                if (!empty($emails)) {
                    $emailsArray = explode(',', $emails);
                    foreach ($emailsArray as $email) {
                        $admin = User::make(['email' => $email]);
                        if ($admin) {
                            $admin->notify(new SendUserDeactivationEmailToAdmin($user));
                        } else {
                            $admin->warn("User with email $email not found.");
                        }
                    }
                }

                session()->flash('success', __('user.firm_user_subscription_cancel_success'));
                session()->flash('reload', 1);
            } catch (Exception $e) {
                session()->flash('error', $e->getMessage());
            }
            // Redirect to the same page (or any other page) and handle the delay in the view

            return redirect()->route('subscriptions.index');
        } else {
            // Subscription not found, return error response
            return redirect()->route('subscriptions.index')->with("error", 'Subscription not found');
        }

    }

    function checkIfPromocodeExists($promocode){
        $stripe = new \Stripe\StripeClient(env('STRIPE_SECRET_KEY'));
        $promotionCodes = $stripe->promotionCodes->all(["code"=> $promocode]);

        if(!$promotionCodes->isEmpty()){
            $promocodeKey = $promotionCodes->data[0];
            return $promocodeKey;
        }else{
            return response()->json('Invalid promocode.', 400);
        }
    }
}
