<?php

namespace App\Http\Controllers\Front;

use App\Firm;
use App\News;
use App\Page;
use App\QREvents;
use App\User;
use App\Content;
use App\Faculty;
use App\Releases;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Rules\CaseSensitiveExists;
use App\Http\Controllers\Controller;
use App\Member;
use App\Notifications\UserVerification;
use App\Notifications\FrontUserVerification;
use Illuminate\Contracts\Auth\PasswordBroker;
use App\Notifications\FrontFirmUserVerification;
use App\QRStatistics;
use Illuminate\Support\Facades\Log;

use function Embed\isEmpty;

class PagesController extends Controller
{
    public function about(){
        $members = Member::where('status',1)->get()->take(siteconfig('about_member_count',0));
        $releases = Releases::where('status',1)->where('released_on', '<=', Carbon::today())->get();
        $pages = Page::all();

        return view('front.pages.about', compact('members','releases', 'pages'));
    }
    public function library(){
        $contents = Content::with('contentFiles', 'speakers')
            ->where('status', 1)
            ->where(function ($query) {
                $query->whereDate('published_on', '<=', Carbon::now())
                      ->orWhereNull('published_on');
            })
            ->orderByDesc('published_on')
            // ->limit(10)
            ->get();

        $showMore = false;
        $pages = Page::all();

        return view('front.pages.library', compact('contents','pages', 'showMore'));
    }

    public function pricing(){
        $pages = Page::all();

        return view('front.pages.pricing', compact('pages'));
    }

    public function faculty(){
        $pages = Page::all();
        $faculty = Faculty::where('status',1)->orderBy('display_order')->get();

        return view('front.pages.faculty', compact('pages', 'faculty'));
    }

    public function facultyShow($slug){
        $faculty = Faculty::where('slug',$slug)->firstOrFail();

        $podcastsData = [];

        if (!empty($faculty->podcasts)) {
            foreach ($faculty->podcasts as $podcast) {
                $podcastsData[] = $podcast->podcast($podcast->podcast_id);
            }
        }

        $contents = collect($podcastsData);

        $categories = $contents->pluck('category')->filter(function ($value) {
            return !is_null($value) && $value !== '';
        })->unique()->values();

        return view('front.pages.faculty_detail', compact('faculty', 'contents', 'categories'));
    }

    public function podcastShow($encodedTitle)
    {
        $podcast = Content::where('title_crc32', $encodedTitle)->first();

        if (!$podcast) {
            $decodedTitle = base64url_decode($encodedTitle);

            $podcast = Content::where('podcast_title', $decodedTitle)->firstOrFail();
        }
        $speakersData = $podcast->speakers->pluck('speaker')->toArray() ?? [];
        return view('front.pages.podcast_detail', compact('podcast', 'speakersData'));
    }

    public function press() {
        $pages = Page::all();
        $news = News::where('status', 1)->get();
    
        $archiveMonths = News::selectRaw('YEAR(created_at) as year, MONTH(created_at) as month, COUNT(*) as count')
            ->where('status', 1)
            ->groupBy('year', 'month')
            ->orderBy('year', 'desc')
            ->orderBy('month', 'desc')
            ->get()
            ->map(function ($item) {
                return [
                    'year' => $item->year,
                    'month' => $item->month,
                    'count' => $item->count,
                    'name' => Carbon::create()->month($item->month)->format('F'),
                ];
            });
    
        return view('front.pages.press', compact('pages', 'news', 'archiveMonths'));
    }
    public function filterNews(Request $request) {
        $query = News::query()->where('status', 1);
    
        if ($request->filled('month_year')) {
            [$year, $month] = explode('-', $request->month_year);
            $query->whereYear('created_at', $year)->whereMonth('created_at', $month);
        }
    
        $news = $query->get();
    
        return response()->json([
            'success' => true,
            'data' => view('front.includes.newslist', compact('news'))->render(),
        ]);
    }
    public function pressDetail($slug) {
        $news = News::where('status', 1)->where('slug',$slug)->firstOrFail();
        return view('front.pages.press_detail', compact('news'));
    }

    public function subscribe(Request $request, $uuid=null){
        $pages = Page::all();

        if($uuid != null){
            if ($uuid) {
                $event = QREvents::where('uuid', $uuid)->where('status',1)->first();
                if ($event) {

                    $userAgent = $request->header('User-Agent'); 
                    $ipAddress = $request->getClientIp();

                    $browser = $this->detectBrowser($userAgent);
                    $platform = $this->detectPlatform($userAgent);
                    $key = $uuid.$ipAddress.$browser;

                    $userStats = QRStatistics::where('session_key',$key)->first();

                    if($userStats){
                        $userStats->update(['client_user_agent'=>$userAgent,
                        'client_browser'=>$browser,
                        'client_ipaddress' => $ipAddress,
                        'client_platform' => $platform,
                        'scan_attempts'=>$userStats->scan_attempts+1, 
                        'updated_at' => now(),
                    ]);
                    }else{
                        QRStatistics::create([
                            'event_id'=>$event->id,
                            'event_uuid'=>$event->uuid,
                            'client_user_agent'=>$userAgent,
                            'client_browser'=>$browser,
                            'client_ipaddress' => $ipAddress,
                            'client_platform' => $platform,
                            'scan_attempts'=>1,
                            'session_key'=>$key,
                            'created_at' => now(),
                        ]);
                    }
                }
            }
        }

        return view('front.pages.subscribe', compact('pages'));
    }

    protected function submitSubscription(Request $request)
    {
        $request = $this->stripHtmlTags($request, User::$notStripTags);
        $data = $request->all();

        // Strip formatting from phone number
        if (!empty($data['phone'])) {
            $data['phone'] = preg_replace('/\D/', '', $data['phone']);
        }

        $this->validation($request);
        $user = User::where('email', $request->get('email'))->first();

        if(empty($data['user_type_id'])){
            $data['user_type_id'] = 3;
        }
        // dd(!empty($user->id));
        if (!empty($user->id)) {
            // Restore user account if already exist
            $user->restore();

            // Update user data.
            $data['deleted_at'] = null;
            $data['password'] = null;
            $data['email_verified_at'] = null;
            $data['status'] = 0;
            if(isset($data['birthdate']) && !empty($data['birthdate'])){
                $data['birthdate'] = Carbon::parse($data['birthdate'])->format('Y-m-d');
            }
            $user->fill($data);
            $user->save();
            $message = 'user.restored_success';
        } else {
            // Save the User Data
            $user = new User;

            $data['password'] = bcrypt($data['password']);

            if(isset($data['birthdate']) && !empty($data['birthdate'])){
                $data['birthdate'] = Carbon::parse($data['birthdate'])->format('Y-m-d');
            }
            $user->fill($data);
            $user->save();

            if($user->user_type_id == 3){
                $firmData = [
                    'name' => "",
                    'address1' => Null,
                    'address2' => Null,
                    'city' => Null,
                    'state_id' => 0,
                    'zipcode' => Null,
                ];
                Firm::updateOrCreate(
                    ['user_id' => $user->id],
                    $firmData
                );
                $message = 'user.student_create_success';
            }
            else{
                $message = 'user.create_success';
            }
        }

        $token = app(\Illuminate\Auth\Passwords\PasswordBroker::class)->createToken($user);
        $user->remember_token = $token;
        $user->save();




        // Send Verification Email to User.
        if($user->user_type_id == 3){
            $firmData = [
                'name' => "",
                'address1' => Null,
                'address2' => Null,
                'city' => Null,
                'state_id' => 0,
                'zipcode' => Null,
            ];
            Firm::updateOrCreate(
                ['user_id' => $user->id],
                $firmData
            );
            if(isset($user->firmcode)){
                $firm = Firm::where('code', $user->firmcode)->first();

                $user->notify(new FrontFirmUserVerification($token, $firm->name));
            }
            else{
                $user->notify(new FrontUserVerification($token));
            }
        }
        else{
            $user->notify(new UserVerification($token));
        }

        return response()->json( ['message' =>'Your account has been created. To continue, please check your email and verify your account.']);
    }

    protected function subscribeSuccess(){
        return view('front.modules.subscribe.success');
    }

    protected function subscribeFailed(){
        return view('front.modules.subscribe.fail');
    }


    private function validation($request)
    {
        //get and use of minimum charctor value and password strength from site config
        $passwordValidationRule = ['required'];
        $passwordPolicy = getPasswordPolicy($passwordValidationRule);

        $rules = [
            'first_name' => 'required',
            'last_name' => 'required',
            'email' => ['required', 'email:rfc,dns', Rule::unique('users')->whereNull('deleted_at')],
            'password' => $passwordPolicy['passwordValidationRule'],
            'confirm_password' => 'same:password',
            'firmcode' => ['nullable', new CaseSensitiveExists('firms', 'code')]
        ];

        $search = array("<span>", "</span>");
        $replace = array("", "  ");
        $newmsg = str_replace($search, $replace,trim( __('user.password_rule_'.$passwordPolicy['passwordStrength'].'_msg', array('min'=>$passwordPolicy['passwordMinLength']))));

        $messages = [
            'confirm_password.same' => 'The password and confirm password must match.',
            'password.regex' => $newmsg
        ];

        $this->validate($request, $rules, $messages);
    }

    public function privacy(){
        $pages = Page::all();

        return view('front.pages.privacy', compact('pages'));
    }

    public function terms(){
        $pages = Page::all();

        return view('front.pages.terms', compact('pages'));
    }

    public function pressRelease(){
        $pages = Page::all();

        return view('front.pages.press_release', compact('pages'));
    }

    public function more(){
        return view('front.pages.more');
    }

    private function detectBrowser($userAgent)
    {
        $browsers = [
            'Edge' => 'Edg/([0-9\.]+)',
            'Opera' => 'OPR/([0-9\.]+)',
            'Chrome' => 'Chrome/([0-9\.]+)',
            'Firefox' => 'Firefox/([0-9\.]+)',
            'Safari' => 'Version/([0-9\.]+).*Safari',
            'Internet Explorer' => 'MSIE ([0-9\.]+)|Trident/.*rv:([0-9\.]+)',
        ];
    
        foreach ($browsers as $browser => $pattern) {
            if (preg_match("~$pattern~i", $userAgent, $matches)) {
                $version = $matches[1] ?? ($matches[2] ?? 'Unknown');
                return $browser . ' ' . $version;
            }
        }
    
        return 'Unknown';
    }

    private function detectPlatform($userAgent)
    {
        if (preg_match('/linux/i', $userAgent)) return 'Linux';
        if (preg_match('/macintosh|mac os x/i', $userAgent)) return 'Mac';
        if (preg_match('/windows|win32/i', $userAgent)) return 'Windows';
        if (preg_match('/android/i', $userAgent)) return 'Android';
        if (preg_match('/iphone|ipad|ipod/i', $userAgent)) return 'iOS';

        return 'Unknown';
    }


}
