<?php

namespace App\Http\Controllers\Front;

use App\EmbedTracker;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use App\PodcastStatistics;
use Illuminate\Database\QueryException;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

use Symfony\Component\HttpFoundation\StreamedResponse;

class SharePodcastController extends Controller
{
//SHARE PODCAST 

public function embedPlayer(Request $request)
{
    $validated = $request->validate([
        'data' => 'required',
        'token' => 'required'
    ]);

    // Decode base64 JSON
    $decodedData = json_decode(base64_decode($validated['data']), true);

    if (!$decodedData) {
        Log::error("Invalid data parameter in embed request.");
        abort(400, 'Invalid player configuration.');
    }

    // Extract data variables
    $player_code = $decodedData['player_size'] . $decodedData['player_theme'];

    $playerTypes = [
        '00' => 'minidark',
        '01' => 'minilight',
        '10' => 'bigdark',
        '11' => 'biglight'
    ];
    
    if (!isset($playerTypes[$player_code])) {
        Log::error("Invalid player_code: $player_code");
        abort(400, 'Invalid player configuration.');
    }
    
    $playerType = $playerTypes[$player_code];

    return view("front.modules.player.embed.$playerType", [
        'contentId' => $decodedData['content_id'],
        'contentFileId' => $decodedData['content_file_id'],
        'podcastTitle' => $decodedData['podcast_title'],
        'podcastLength' => $decodedData['podcast_length'],
        'podcastSpeakers' => $decodedData['podcast_speakers'],
        'token' => $validated['token']
    ]);
}

private function detectBrowser($userAgent)
{
    $browsers = [
        'Edge' => 'Edg/([0-9\.]+)',
        'Opera' => 'OPR/([0-9\.]+)',
        'Chrome' => 'Chrome/([0-9\.]+)',
        'Firefox' => 'Firefox/([0-9\.]+)',
        'Safari' => 'Version/([0-9\.]+).*Safari',
        'Internet Explorer' => 'MSIE ([0-9\.]+)|Trident/.*rv:([0-9\.]+)',
    ];

    foreach ($browsers as $browser => $pattern) {
        if (preg_match("~$pattern~i", $userAgent, $matches)) {
            $version = $matches[1] ?? ($matches[2] ?? 'Unknown');
            return $browser . ' ' . $version;
        }
    }

    return 'Unknown';
}

public function trackEmbedCode(Request $request)
{
    try {
        $userAgent = $request->header('User-Agent'); 
        $browser = $this->detectBrowser($userAgent);
        $podcastkey = $request->input('contentId').$request->input('contentFileId').$request->input('websiteUrl').$request->input('token');

        $existingpsEntry = PodcastStatistics::where('key', ($podcastkey))->first();
        if (!$existingpsEntry) {
            PodcastStatistics::create([
                'content_id' => $request->input('contentId'),
                'content_file_id' => $request->input('contentFileId'),
                'site_name' => $request->input('websiteUrl'),
                'browser' => $browser,
                'ipaddress' => $request->input('ipAddress'),
                'token' => $request->input('token'),
                'key'=>($podcastkey)
            ]);
            return response()->json(['message' => 'Tracker created'], 201);
        }else{
            return response()->json(['message' => 'Entry already exists with this URL and token.'], 201);
        }

    } catch (QueryException $e) {
        if ($e->getCode() == 23000) {
            return response()->json(['message' => 'Entry already exists with this URL and token.'], 409);
        }

        Log::error('Database error: ' . $e->getMessage());
        return response()->json(['message' => 'Database error, please try again.'], 500);
    }
}

public function trackPodcastPlay(Request $request)
{
    try {
        $existingpsEntry = PodcastStatistics::where('key', $request->input('key'))->first();
        if ($existingpsEntry) {
            switch ($request->input('deviceType')) {
                case 'android':
                    $existingpsEntry->increment('android', 1);
                    break;
                case 'iPhone':
                    $existingpsEntry->increment('iphone', 1);
                    break;
                case 'web':
                    $existingpsEntry->increment('web', 1);
                    break;
                default:
                    break;
            }
            
        }
    } catch (QueryException $e) {
        if ($e->getCode() == 23000) {
            return response()->json(['message' => 'Entry already exists with this URL and token.'], 409);
        }

        Log::error('Database error: ' . $e->getMessage());
        return response()->json(['message' => 'Database error, please try again.'], 500);
    }
}


public function shareConfig(Request $request)
{
    $validated = $request->validate([
        'content_id' => 'required|integer',
        'content_file_id' => 'required|integer',
        'player_size' => 'required|string',
        'player_theme' => 'required|string',
        'podcast_title' => 'required',
        'podcast_length' => 'required',
        'podcast_speakers' => 'required',
        'file' => 'required',
    ]);


    $filePath = "upload_file/" . $validated['file'];

    // Check if file exists
    if (!Storage::disk()->exists($filePath)) {
        return abort(404, "Audio file not found.");
    }

    // Get file URL
    $token = Crypt::encryptString($filePath);

    $data = [
        'content_id' => $validated['content_id'],
        'content_file_id' => $validated['content_file_id'],
        'player_size' => $validated['player_size'],
        'player_theme' => $validated['player_theme'],
        'podcast_title' => $validated['podcast_title'],
        'podcast_length' => $validated['podcast_length'],
        'podcast_speakers' => $validated['podcast_speakers']
    ];


    $encodedData = base64_encode(json_encode($data));

    // Generate Embed URL
    $embedUrl = route('podcast.embed', [
        'data' => $encodedData,
        'token' => $token
    ]);

        // Extract data variables
    $player_code = $data['player_size'] . $data['player_theme'];

    $playerTypes = [
        '00' => 'minidark',
        '01' => 'minilight',
        '10' => 'bigdark',
        '11' => 'biglight'
    ];
    
    if (!isset($playerTypes[$player_code])) {
        Log::error("Invalid player_code: $player_code");
        abort(400, 'Invalid player configuration.');
    }
    
    $playerType = $playerTypes[$player_code];

    $previewImage = "assets/player/assets/preview/{$playerType}.png";

    $height = $data['player_size'] == 1 ? 700 : 250;

    $iframeCode = "<iframe src='{$embedUrl}' width='100%' height='{$height}' frameborder='0' allowfullscreen></iframe>";

    return response()->json([
        'message' => 'Player configuration saved successfully!',
        'embed_code' => $iframeCode,
        'previewImage'=>$previewImage,
    ]);
}

public function serveAudio($token)
{

try {
    // Decrypt token to get the file path
    $filePath = Crypt::decryptString($token);

    // Check if file exists
    if (!Storage::disk()->exists($filePath)) {
        return abort(404, "Audio file not found.");
    }

    // Get the full path of the file
    $fullPath = Storage::disk()->path($filePath);
    
    // Get file size
    $fileSize = Storage::disk()->size($filePath);

    // Handle local environment
    if (App::environment('local')) {
        return response()->file($fullPath, [
            'Content-Type' => 'audio/mp3',
            'Content-Disposition' => 'inline',
            'Accept-Ranges' => 'bytes',
            'Content-Length' => $fileSize,
            'Content-Range' => "bytes 0-{$fileSize}/{$fileSize}",
        ]);
    }

    // Generate a signed URL for cloud storage (valid for 1 hour)
    $signedUrl = Storage::disk()->temporaryUrl($filePath, now()->addHour());

    return new StreamedResponse(function () use ($signedUrl) {
        $stream = fopen($signedUrl, 'r');
        fpassthru($stream);
        fclose($stream);
    }, 200, [
        'Content-Type' => 'audio/mp3',
        'Content-Disposition' => 'inline',
        'Accept-Ranges' => 'bytes',
        'Content-Length' => $fileSize,
        // 'Content-Range' => "bytes 0-{$fileSize}/{$fileSize}",
        'Pragma'=>"public",
        'Expires'=>'0',
        'X-Pad'=>'avoid browser bug',
    ]);

} catch (\Exception $e) {
    Log::error("Error streaming audio: " . $e->getMessage());
    return abort(500, "Unable to process the request.");
}

    
}


public function serveAudioForWeb($filePath)
{
try {
    // Decrypt token to get the file path
    $file = Crypt::decryptString($filePath);
    $filePath = "upload_file/" . $file;

    // Check if file exists
    if (!Storage::disk()->exists($filePath)) {
        return abort(404, "Audio file not found.");
    }

    // Get the full path of the file
    $fullPath = Storage::disk()->path($filePath);
    
    // Get file size
    $fileSize = Storage::disk()->size($filePath);

    // Handle local environment
    if (App::environment('local')) {
        return response()->file($fullPath, [
            'Content-Type' => 'audio/mp3',
            'Content-Disposition' => 'inline',
            'Accept-Ranges' => 'bytes',
            'Content-Length' => $fileSize,
            'Content-Range' => "bytes 0-{$fileSize}/{$fileSize}",
        ]);
    }

    // Generate a signed URL for cloud storage (valid for 1 hour)
    $signedUrl = Storage::disk()->temporaryUrl($filePath, now()->addHour());

    return new StreamedResponse(function () use ($signedUrl) {
        $stream = fopen($signedUrl, 'r');
        fpassthru($stream);
        fclose($stream);
    }, 200, [
        'Content-Type' => 'audio/mp3',
        'Content-Disposition' => 'inline',
        'Accept-Ranges' => 'bytes',
        'Content-Length' => $fileSize,
        // 'Content-Range' => "bytes 0-{$fileSize}/{$fileSize}",
        'Pragma'=>"public",
        'Expires'=>'0',
        'X-Pad'=>'avoid browser bug',
    ]);

} catch (\Exception $e) {
    Log::error("Error streaming audio: " . $e->getMessage());
    return abort(500, "Unable to process the request.");
}

    
}
}