<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Contracts\Routing\ResponseFactory;
use Illuminate\Http\JsonResponse;
use App\SiteConfig;

class ApiResponse
{
    /**
     * The Response Factory our app uses
     *
     * @var ResponseFactory
     */
    protected $factory;

    /**
     * JsonMiddleware constructor.
     *
     * @param ResponseFactory $factory
     */
    public function __construct(ResponseFactory $factory)
    {
        $this->factory = $factory;
    }

    /**
     * Handle an incoming request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Closure  $next
     * @return mixed
     */
    public function handle($request, Closure $next)
    {
        $request->headers->set('Accept', 'application/json');
        $request->headers->set('Access-Control-Allow-Origin', '*');
        $request->headers->set('device-type', $request->header('device-type'));
        $request->headers->set('application-version', $request->header('application-version'));


        if (!in_array($request->header('device-type'),['ios','web','android'])) {
            $data = [];
            $data['message'] = trans('common.notfound_devicetype_msg');
            return response()->json( $data, 422);
        }
        if (in_array($request->header('device-type'),['ios','web','android'])) {
            $version = $request->header('application-version');
            $device_type = $request->header('device-type');
            $data = [];
            if ($device_type == 'ios') {
                if (!version_compare(siteconfig('ios_app_version'), $version, '<=')) {
                    $data['message'] = trans('New ios app version is available. Please update to the latest version of the app for a better experience.');
                    return response()->json( $data, 426);
                }
            } elseif ($device_type == 'android') {
                if (!version_compare(siteconfig('android_app_version'), $version, '<=')) {
                    $data['message'] = trans('New android app version is available. Please update to the latest version of the app for a better experience.');
                    return response()->json( $data, 426);
                }
            } elseif ($device_type == 'web') {
                if (!version_compare(siteconfig('web_version'), $version, '<=')) {
                    $data['message'] = trans('New web version is available. Please update to the latest version of the app for a better experience.');
                    return response()->json( $data, 426);
                }
            }

        }


        // Get the response
        $response = $next($request);

        //not found condition
        if ($response->getStatusCode() == 404) {
            $data = [];
            $data['message'] = trans('common.exception_msg');
            return response($data);
        }
        if ($response instanceof JsonResponse) {
            $this->setJsonResponse($response);
        } elseif ($response instanceof \Illuminate\Http\RedirectResponse) {
            $this->setRedirectResponse();
        } else {
            if ($response->getOriginalContent() != null) {
                $data = $response->getOriginalContent()->getData();
                $data = json_decode(json_encode($data), true);
                array_walk_recursive($data, function (&$item) {
                    if (is_null($item)) {
                        $item=strval($item);
                    }
                });
                $response->setContent($data);
            }
        }

        if (!empty($response->getStatusCode()) && $response->getStatusCode() == 422) {
            $data = $response->getData();
            $data->message = $this->setErrorMessage($data);
            $response->setData($data);
        }

        return $response;
    }

    /** Remove the cognitive Complexity for the Json Response */
    private function setJsonResponse($response){
        if (property_exists($response->getData(), 'exception')) {
            $exception = $response->getData()->exception;
            if (!empty($exception)) {
                $data = [];
                $data['message'] = trans('common.exception_msg');
                return response($data);
            }
        }
    }

    /** Remove the cognitive Complexity for the Redirection Response */
    private function setRedirectResponse(){
        $data = [];
        $data = session()->all();
        if (isset($data['status']) && !empty($data['status'])) {
            $data['message'] = $data['status'];
        }
        if (isset($data['success']) && !empty($data['success'])) {
            $data['message'] = $data['success'];
        }
        if (isset($data['error']) && !empty($data['error'])) {
            $data['message'] = $data['error'];
        }
        return response($data);
    }

    /** Set all the error message of the API Response */
    private function setErrorMessage($data){
        $data->message = '';
        if (!empty($data->errors)) {
            foreach ($data->errors as $error) {
                $data->message .= implode(PHP_EOL, $error) . PHP_EOL;
            }
        }
        return rtrim($data->message, PHP_EOL);
    }

}
