<?php

namespace App;

use App\Traits\DbEvents;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

use function PHPSTORM_META\type;

class PodcastStatistics extends Model
{
    use HasFactory , DbEvents;
    protected $table = "podcast_statistics";
    
    protected $fillable = [
        'content_id',
        'content_file_id',
        'android',
        'iphone',
        'web',
        'home_page',
        'library',
        'site_name',
        'browser',
        'ipaddress',
        'token',
        'key'
    ];

    public function content(){
        return $this->belongsTo(Content::class,'content_id')->with('speakers');
    }

    public function calculateTotalPlays(){
        return PodcastStatistics::where('content_id', $this->content_id)
        ->sum(DB::raw('android + iphone + web + home_page + library'));
    }

    public function getResult($request)
    {    
        // Set default parameter values
        $order_by = !empty($request->get('order_by')) ? $request->get('order_by') : 'id';
        $order = !empty($request->get('order')) ? $request->get('order') : 'asc';
        $defaultper_page = !empty($request->get('per_page')) ? $request->get('per_page') : siteconfig('records_per_page', 0);

        // Fetch stats list
        $stats = PodcastStatistics::with('content')->groupBy('content_id');

        // Search
        if (!empty($request->get('search'))) {
            $searchStr = gettype($request->get('search')) == 'string' ? $request->get('search') : $request->get('search')['value'] ;
            $escape = "ESCAPE '|'";
             if(substr_count($searchStr,"|")){
                $searchStr = str_replace('\\', '\\\\\\', $searchStr);
                 $escape = "";
             }
            // added escape for searching backslash issue DLC-140
            $stats = $stats->whereHas('content', function ($query) use ($searchStr) {
                $query->where('podcast_title', 'LIKE', "%{$searchStr}%")
                      ->orWhere('internal_cl_podcast_id', 'LIKE', "%{$searchStr}%");
            });
        }

        // published_on
        if ($request->get('published_on') !== null) {
            $date = $request->get('published_on');
            $date = \Carbon\Carbon::createFromFormat('m/d/Y', $date)->format('Y-m-d');
            $stats = $stats->whereHas('content', function ($query) use ($date) {
                $query->whereDate('published_on', $date);
            });
            
        }
        //if per_page not set then all record value use for per_page
        // Curretly it is not in Admin user listing as we are using data table paging
        // but have kept this for API integration
        $per_page = !empty($defaultper_page) ? $defaultper_page : $stats->count();

        // Order By & Pagination
        $stats = $stats->paginate($per_page);

        return $stats;
    }
}
