<?php

namespace App;

use Cache;
use App\Traits\DbEvents;
use Illuminate\Database\Eloquent\Model;

class SiteConfig extends Model
{
    use DbEvents;

    /**
     * Overwrite created_by field value with currently logged in user.
     * Set @var has_created_by to false if created_by field does not exist in DB Table.
     *
     * @var boolean
     */
    protected $has_created_by = true;

    /**
     * Overwrite updated_by field value with currently logged in user.
     * Set @var has_updated_by to false if created_by field does not exist in DB Table.
     *
     * @var boolean
     */

    protected $has_updated_by = true;

    /**
     * Cached variables
     * @var array
     */
    protected $cache = ['site_config.all'];

    /**
     * Define feilds name which have html tags
     * Set @var notStripTags add field name which field value have html tags.
     *
     * @var array
     */

    public static $notStripTags = ['email','from_email','admin_email','robots_meta_tag','google_analytics_code','google_tag_manager_code','googlemap_embedded','brand_product_content'];

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'name',
        'val',
        'type',
        'status',
        'created_by',
        'updated_by'
    ];

    /**
     * Get all the site configs
     *
     * @return mixed
     */
    public static function getAllSiteConfigs()
    {
        return Cache::rememberForever('site_config.all', function () {
            return self::all();
        });
    }

    /**
     * Get a site config value
     *
     * @param $key
     * @param null $default
     * @return bool|int|mixed
     */
    public static function get($key, $default = null)
    {
        if (self::has($key)) {
            $setting = self::getAllSiteConfigs()->where('name', $key)->first();
            return self::castValue($setting->val, $setting->type);
        }

        return self::getDefaultValue($key, $default);
    }

    /**
     * Check if setting exists
     *
     * @param $key
     * @return bool
     */
    public static function has($key)
    {
        return (boolean) self::getAllSiteConfigs()->whereStrict('name', $key)->count();
    }

    /**
     * Get default value from config if no value passed
     *
     * @param $key
     * @param $default
     * @return mixed
     */
    private static function getDefaultValue($key, $default)
    {
        return is_null($default) ? self::getDefaultValueForField($key) : $default;
    }

    /**
     * Get default value for a site config
     *
     * @param $field
     * @return mixed
     */
    public static function getDefaultValueForField($field)
    {
        return self::getDefinedSiteConfigFields()
                ->pluck('value', 'name')
                ->get($field);
    }

    /**
     * Get all the site config fields from config
     *
     * @return Collection
     */
    protected static function getDefinedSiteConfigFields($module = '')
    {
        return collect(config('siteconfig'))->filter(function ($value) use ($module) {
            if (!empty($module) && !empty($value['module']) && $value['module'] != $module) {
                return false;
            }

            return true;
        })->pluck('inputs')->flatten(1);
    }

    /**
     * caste value into respective type
     *
     * @param $val
     * @param $castTo
     * @return bool|int
     */
    private static function castValue($val, $castTo)
    {
        switch ($castTo) {
            case 'int':
            case 'integer':
                return intval($val);
                break;

            case 'bool':
            case 'boolean':
                return boolval($val);
                break;

            default:
                return $val;
        }
    }

    /**
     * Get the validation rules for site config fields
     *
     * @param string $module
     * @return array
     */
    public static function getValidationRules($module)
    {
        return self::getDefinedSiteConfigFields($module)->pluck('rules', 'name')->reject(function ($val) {
            return is_null($val);
        })->toArray();
    }

    /**
     * Get the data type of a stie config
     *
     * @param $field
     * @return mixed
     */
    public static function getDataType($field)
    {
        $type  = self::getDefinedSiteConfigFields()
                ->pluck('data', 'name')
                ->get($field);

        return is_null($type) ? 'string' : $type;
    }
}
