<?php

namespace App;

use App\Exports\SubmissionExport;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use App\Model;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Facades\Excel;
use stdClass;

class Submissions extends Model
{
    use HasFactory;

    /**
     * Overwrite created_by field value with currently logged in user.
     * Set @var has_created_by to false if created_by field does not exist in DB Table.
     *
     * @var boolean
     */
    protected $has_created_by = true;

    /**
     * Overwrite updated_by field value with currently logged in user.
     * Set @var has_updated_by to false if created_by field does not exist in DB Table.
     *
     * @var boolean
     */

    protected $has_updated_by = true;

    public static $notStripTags = [];

    protected $appends = ['content_name', 'content_file_name', 'image_path','certification_download_url', 'provider_name'];

    protected $dates = [
        'exported_at',
        'submitted_at',
    ];

    protected $fillable = [
        'user_id',
        'bar_id',
        'content_id',
        'content_file_id',
        'content_state_id',
        'credit_submitted',
        'subscription_id',
        'stripe_subscription_id',
        'created_by',
        'updated_by',
        'status',
        'exported_at',
        'submitted_at',
        'approved_by',
        'approved_at',
    ];


    public function getExportedAtAttribute($value)
    {
        if($value){
            return Carbon::parse($value)->format(config('app.db_date_time_short_format'));
        }else{
            "";
        }

    }

    public function getSubmittedAtAttribute($value)
    {
        if($value){
            return Carbon::parse($value)->format(config('app.db_date_time_short_format'));
        }else{
            "";
        }
    }


    public function user(){
        return $this->belongsTo('App\User', 'user_id', 'id');
    }

    public function getContentWhereId($id)
    {
        $content=Content::where('id',$id)->first();
        return $content;
    }

    public function getContentFileWhereId($id,$content_id)
    {
        $contentFiles=ContentFiles::where('id',$id)->where('contents_id',$content_id)->first();
        return $contentFiles;
    }

    public function getBarWhereId($id)
    {
        $bar=Bar::where('id',$id)->first();
        return $bar;
    }

    public function getContentStateWhereBarId($bar_id, $content_id)
    {
        $contentState=ContentsStates::where('bar_id',$bar_id)->where('contents_id',$content_id)->first();
        return $contentState;
    }

    public function getBarUserRefWhereBarId($bar_id, $user_id)
    {
        $barUserRef=BarUserReference::where('bar_id',$bar_id)->where('user_id', $user_id)->first();
        return $barUserRef;
    }

    public function getResult($request)
    {
        // Set default parameter values
        $order_by = !empty($request->get('order_by')) ? $request->get('order_by') : 'submissions.id';
        $order = !empty($request->get('order')) ? $request->get('order') : 'asc';

        // Fetch Submissions list

        $submissions = new Submissions;

        // Search
        if (!empty($request->get('search'))) {
            $searchStr = $request->get('search');
            $escape = "ESCAPE '|'";
            if(substr_count($searchStr,"|")){
            $searchStr = str_replace('\\', '\\\\\\', $searchStr);
                $escape = "";
            }

            $nameParts = explode(' ', $searchStr);
        // added escape for searching backslash issue DLC-140
            $submissions = $submissions->select('users.*','contents.*','contents_states.*','bars.*','bar_user_references.*','submissions.*','submissions.status as status');
            $submissions = $submissions
            ->join('users', 'submissions.user_id', '=', 'users.id')
            ->join('contents', 'submissions.content_id', '=', 'contents.id')
            ->join('contents_states', 'submissions.content_state_id', '=', 'contents_states.id')
            ->join('bars', 'submissions.bar_id', '=', 'bars.id')
            ->join('bar_user_references', function ($join) {
                $join->on('submissions.bar_id', '=', 'bar_user_references.bar_id')
                    ->on('submissions.user_id', '=', 'bar_user_references.user_id');
            })
            ->where(function ($query) use ($searchStr, $escape,$nameParts) {
                $query
                    ->where(function ($innerQuery) use ($searchStr, $escape,$nameParts) {
                        foreach ($nameParts as $part) {
                            $innerQuery->orWhere(function ($q) use ($part, $escape) {
                                $q->whereRaw('users.first_name LIKE ? '. $escape, '%' . $part . '%')
                                  ->orWhereRaw('users.last_name LIKE ? '. $escape, '%' . $part . '%');
                            });
                        }
                    })
                    ->orWhereRaw('contents.podcast_title LIKE ? '.$escape, '%'.$searchStr.'%')
                    ->orWhereRaw('bars.state_name LIKE ? '.$escape, '%'.$searchStr.'%')
                    ->orWhereRaw('contents_states.state_bar_id LIKE ? '.$escape, '%'.$searchStr.'%')
                    ->orWhereRaw('bar_user_references.bar_number LIKE ? '.$escape, '%'.$searchStr.'%')
                    ->orWhereRaw('submissions.credit_submitted LIKE ? '.$escape, '%'.$searchStr.'%');
            });
        }

         // Filter by firm
         if ($request->get('firm') !== null) {
            $submissions = $submissions->whereHas('user', function ($query) use ($request) {
                $query->where('firmcode', $request->get('firm'));
            });
        }

        if ($request->get('status') !== null) {
            $submissions = $submissions->where('submissions.status', $request->get('status'));
        }
        if ($request->get('id') !== null) {
            $submissions = $submissions->whereIn('submissions.id', $request->get('id'));
        }

        // Filter
        if ($request->get('type') !== null) {
            $type = $request->get('type');
            $today = now()->format('Y-m-d');


            if ($type == 1) {
                //this month
                $submissions = $submissions->whereYear('submissions.created_at', now()->year)->whereMonth('submissions.created_at', now()->month);
            }else if ($type == 2) {
                //last month
                $lastMonthStart = now()->subMonth()->startOfMonth()->format('Y-m-d');
                $lastMonthEnd = now()->subMonth()->endOfMonth()->format('Y-m-d');
                $submissions = $submissions->whereDate('submissions.created_at', '>=', $lastMonthStart)->whereDate('submissions.created_at', '<=', $lastMonthEnd);
            }else if ($type == 3) {
                //date range
                if ($request->has('start_date') && $request->has('end_date')) {
                    $start_date = $request->input('start_date');
                    $end_date = $request->input('end_date');

                    $submissions = $submissions->whereDate('submissions.created_at', '>=', \Carbon\Carbon::parse($start_date)->format('Y-m-d'))->whereDate('submissions.created_at', '<=', \Carbon\Carbon::parse($end_date)->format('Y-m-d'));
                }
            }else if ($type == 4) {
                //till date (all)
                $submissions = $submissions->whereDate('submissions.created_at', '<=', $today);
            }else if ($type == 5) {
                //this year
                $currentYear = now()->year;
                $submissions = $submissions->whereYear('submissions.created_at', $currentYear);
            }
        }else{
            $submissions = $submissions->whereYear('submissions.created_at', now()->year)->whereMonth('submissions.created_at', now()->month);
        }

       

        if (Auth::user()->user_type_id == 2) {
            $auth_user = Auth::user();
            $user_ids = User::where('firmcode', $auth_user->firm->code)->pluck('users.id')->toArray();
            $submissions = $submissions->whereIn('submissions.user_id', $user_ids);
        }

        //if per_submissions not set then all record value use for per_submissions
        // Curretly it is not in Admin user listing as we are using data table paging
        // but have kept this for API integration
        // $per_submissions = !empty($defaultper_submissions) ? $defaultper_submissions : $submissions->count();

        $per_page = !empty($request->get('per_page')) ? $request->get('per_page') : $submissions->count();


        // Order By & Pagination
        $submissions = $submissions->orderBy($order_by, $order)->paginate($per_page);

        return $submissions;
    }


    public function getReportData(){
        $submissionModel = new Submissions;
        $today = now()->format('Y-m-d');
        $submissions=$submissionModel->whereDate('submissions.created_at', $today)->get();
        $details=[];

        foreach ($submissions as $submission) {

            // Create a new stdClass object to store details
            $submissionDetails = new stdClass();

            // Set properties of the object
            $submissionDetails->general=0;
            $submissionDetails->ethics=0;
            $submissionDetails->dual=0;
            $submissionDetails->id = $submission->id;
            $submissionDetails->user_id = $submission->user_id;
            $submissionDetails->bar_id = $submission->bar_id;
            $submissionDetails->content_id = $submission->content_id;
            $submissionDetails->content_file_id = $submission->content_file_id;

            // get user bar id
            $submissionDetails->user_bar_id = $submission->getBarUserRefWhereBarId($submission->bar_id, $submission->user_id)->bar_number;

            // get user name
            $submissionDetails->user_name = $submissionDetails->user_name = $submission->user->getFullNameWhereId($submission->user_id);
            // get bar name
            $submissionDetails->state_name = $submission->getBarWhereId($submission->bar_id)->state_name;

            // get podcast bar id
            $submissionDetails->podcast_id_for_bar = $submission->getContentStateWhereBarId($submission->bar_id,$submissionDetails->content_id)->state_bar_id;

            // get podacst  name
            $submissionDetails->podcast_title = $submission->getContentWhereId($submissionDetails->content_id)->podcast_title;

            $submissionDetails->podcast_bar_id = $submission->getContentFileWhereId($submissionDetails->content_file_id, $submissionDetails->content_id)->bar_id;

            // if (Auth::user()->user_type_id == 1) {
            //     // get firm name in case of super admin
            //     $userFirmCode = User::select('firmcode')->where('id',$submissionDetails->user_id)->first()->firmcode;
            //     $submissionDetails->firm_name = Firm::select('name')->where('code', $userFirmCode)->first()->name;
            // }

            // get credit submitted
            $submissionDetails->credits_submitted = $submission->credit_submitted;

            //get submitted date
            $submissionDetails->created_at = $submission->created_at;

            //get general/ethics/dual hours
            $submissionDetails->content_type = $submission->getContentWhereId($submissionDetails->content_id)->cl_type;
            if($submissionDetails->content_type == 1){
                $submissionDetails->general = $submissionDetails->credits_submitted;
            } elseif($submissionDetails->content_type == 2){
                $submissionDetails->ethics = $submissionDetails->credits_submitted;
            }else{
                $submissionDetails->dual = $submissionDetails->credits_submitted;
            }

            $details[] = $submissionDetails;
        }

        return $details;

    }

    public function getReportDataCount(){
        $submissions = $this->getReportData();
        return count($submissions);
    }

    public function getReportDataSum(){
        $submissions = $this->getReportData();
        return array_sum(array_column($submissions, 'credits_submitted'));
    }

    public function userCredits()
    {
        return $this->hasMany(UserCredits::class, 'submission_id', 'id');
    }


    public function getContentNameAttribute()
    {
        $content = $this->belongsTo(Content::class, 'content_id', 'id')->first();
        return $content ? $content->podcast_title : null;
    }

    public function getImagePathAttribute()
    {
        $content = $this->belongsTo(Content::class, 'content_id', 'id')->first();
        return $content ? $content->image_path : null;
    }

    public function getContentFileNameAttribute()
    {
        $content_file = $this->belongsTo(ContentFiles::class, 'content_file_id', 'id')->first();
        return $content_file ? $content_file->file_name : null;
    }

    public function getProviderNameAttribute()
    {
        $content = $this->belongsTo(Content::class, 'content_id', 'id')->first();
        return $content ? $content->provider_name : null;
    }

    public function getCertificationDownloadUrlAttribute()
    {
        return route('downloadCertificate', ['encoded_id' => base64_encode($this->id)]);
    }

}
