<?php

namespace App;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use App\Model;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Auth;

class Subscription extends Model
{
    use HasFactory, SoftDeletes;

    protected $guarded = [];

    protected $dates = [
        'start_date',
        'end_date',
    ];

    public function getStartDateAttribute($value)
    {
        return Carbon::parse($value)->format(config('app.db_date_format'));
    }

    public function getEndDateAttribute($value)
    {
        return Carbon::parse($value)->format(config('app.db_date_format'));
    }

    public function bar(){
        return $this->belongsTo('App\Bar', 'bar_id', 'id');
    }

    public function barUserReference(){
        return $this->belongsTo('App\BarUserReference', 'bar_user_references_id', 'id');
    }

    public function user(){
        return $this->belongsTo('App\User', 'user_id', 'id');
    }


    public function getAll($ignoreExpiredRecords=0){
        $subscriptions = $this->newQuery()->with(['user', 'bar']);
        $subscriptions->whereHas('user', function ($query) {
            $query->whereNull('deleted_at');
        });
        if($ignoreExpiredRecords == 1){
            $subscriptions->whereNot('stripe_subscription_status','expired')->whereNot('status',0);
        }
        if (Auth::user()->user_type_id == 2) {
            $userIds = User::where('firmcode', Auth::user()->firm->code)->pluck('id');
            return $subscriptions->whereIn('user_id', $userIds)->get();
        }
        return $subscriptions->get();
    }

    public function getResult($request,$ignoreExpiredRecords=0)
    {

        $order_by = !empty($request->get('order_by')) ? $request->get('order_by') : 'id';
        $order = !empty($request->get('order')) ? $request->get('order') : 'asc';


        $subscriptions = $this->newQuery()->with(['user', 'bar']);
        $subscriptions->whereHas('user', function ($query) {
            $query->whereNull('deleted_at');
        });
        if ($search = $request->get('search')) {
            $subscriptions->whereHas('user', function ($query) use ($search) {
                $query->where('first_name', 'LIKE', "%{$search}%")
                      ->orWhere('last_name', 'LIKE', "%{$search}%");
            })->orWhereHas('bar', function ($query) use ($search) {
                $query->where('state_name', 'LIKE', "%{$search}%");
            })->orWhere('stripe_subscription_status', 'LIKE', "%{$search}%");
        }

        // Apply filters
        if ($name = $request->get('name')) {
            $subscriptions->whereHas('user', function ($query) use ($name) {
                $query->where('id', $name);
            });
        }
        if ($bar = $request->get('bar')) {
            $subscriptions->where('bar_id', $bar);
        }
        if ($status = $request->get('status')) {
            $subscriptions->where('stripe_subscription_status', 'LIKE', $status);
        }
        if ($firm = $request->get('firm')) {
            $subscriptions->whereHas('user', function ($query) use ($firm) {
                $query->where('firmcode', $firm);
            });
        }

        if (Auth::user()->user_type_id == 2) {
            $userIds = User::where('firmcode', Auth::user()->firm->code)->pluck('id');
            $subscriptions->whereIn('user_id', $userIds);
        }
        if($ignoreExpiredRecords == 1){
            $subscriptions->whereNot('stripe_subscription_status','expired')->whereNot('status',0);
        }

        $per_page = !empty($request->get('per_page')) ? $request->get('per_page') : $subscriptions->count();

        return $subscriptions->orderBy($order_by, $order)->paginate($per_page);
    }
}
