<?php

namespace App\Traits;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\DB;

trait DisplayOrder
{
    public static function bootDisplayOrder()
    {
        static::addGlobalScope('orderByDisplayOrder', function (Builder $builder) {
            $builder->orderBy('display_order', 'asc');
        });

        static::saving(function ($model) {
            if (!isset($model->display_order) || empty($model->display_order)) {
                $tableName = $model->getTable();
                $model->display_order = DB::table($tableName)
                    ->whereNull('deleted_at')
                    ->max('display_order') + 1 ?? 1;
            }
        });

        static::updating(function ($model) {
            $originalOrder = $model->getOriginal('display_order');
            $newOrder = $model->display_order;

            if ($newOrder != $originalOrder) {
                $tableName = $model->getTable();

                // Adjust other records
                if ($newOrder > $originalOrder) {
                    // Shift records up if the new order is higher
                    DB::table($tableName)
                        ->whereBetween('display_order', [$originalOrder + 1, $newOrder])
                        ->whereNull('deleted_at')
                        ->decrement('display_order');
                } elseif ($newOrder < $originalOrder) {
                    // Shift records down if the new order is lower
                    DB::table($tableName)
                        ->whereBetween('display_order', [$newOrder, $originalOrder - 1])
                        ->whereNull('deleted_at')
                        ->increment('display_order');
                }
            }

            if (!isset($model->display_order) || empty($model->display_order)) {
                $tableName = $model->getTable();
                $model->display_order = DB::table($tableName)
                    ->whereNull('deleted_at')
                    ->max('display_order') + 1 ?? 1;
            }
        });
    }
}
