<?php

namespace App;

use Event;
use SoftDeletes;

use Laravel\Passport\HasApiTokens;
use App\Traits\DbEvents;
use Illuminate\Notifications\Notifiable;
use App\Notifications\MailResetPasswordNotification;
use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Foundation\Auth\User as Authenticatable;
use App\UserType;
use Auth;
use Illuminate\Support\Facades\DB;

class User extends Authenticatable implements MustVerifyEmail
{
    use HasApiTokens, Notifiable, SoftDeletes, DbEvents;

    /**
     * Overwrite created_by field value with currently logged in user.
     * Set @var has_created_by to false if created_by field does not exist in DB Table.
     *
     * @var boolean
     */
    protected $has_created_by = true;
    public $profile;
    /**
     * Overwrite updated_by field value with currently logged in user.
     * Set @var has_updated_by to false if created_by field does not exist in DB Table.
     *
     * @var boolean
     */

    protected $has_updated_by = true;

    /**
     * Define feilds name which have html tags
     * Set @var notStripTags add DB Table column name which column have html tags.
     *
     * @var array
     */

    public static $notStripTags = ['email'];



    public function setProfile($data)
    {
        $this->profile = $data;
    }

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'user_type_id',
        'email',
        'securitycode',
        'password',
        'first_name',
        'last_name',
        'profile_image',
        'login_attempt_count',
        'firmcode',
        'subscription_status',
        'phone',
        'company_name',
        'job_title',
        'birthdate',
        'email_verified_at',
        'block',
        'display_name',
        // 'user_firm_name',
        // 'user_firm_address1',
        // 'user_firm_address2',
        // 'user_firm_city',
        // 'user_firm_state_id',
        // 'user_firm_zip',
        'user_work_email',
        'user_work_phone',
        'user_practice_area',
        'is_completed_profile',
        'stripe_id',
        'status',
        'created_by',
        'updated_by'
    ];

    /**
     * The attributes that should be hidden for arrays.
     *
     * @var array
     */
    protected $hidden = [
        'password', 'remember_token',
    ];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
        'birthdate' => 'date',
        'user_practice_area' => 'array',
    ];

    public function scopeActive($query)
    {
        return $query->where('status', 1);
    }
    
    protected $appends = ['user_firm_name','user_firm_address1','user_firm_address2','user_firm_city','user_firm_state_id','user_firm_zip'];

    public function appUserFirm()
    {
        if(empty($this->firmcode)){
            return $this->hasOne('App\Firm', 'user_id', 'id');
        }else{
            return $this->hasOne('App\Firm', 'code', 'firmcode');
        }

    }

    public function firm()
    {
        return $this->hasOne('App\Firm', 'user_id', 'id');
    }

    public function subscriptions()
    {
        return $this->hasMany('App\Subscription', 'user_id', 'id')->where('status',1);
    }

    public function submissions()
    {
        return $this->hasMany('App\Submissions', 'user_id', 'id');
    }

    public function contentListerners()
    {
        return $this->hasMany('App\ContentListener', 'user_id', 'id');
    }

    /**
     * User & User Type Relationship [Many to One]
     * E.g.: Multiple Users are Available with Admin User Role.
     */
    public function userType()
    {
        return $this->belongsTo('App\UserType');
    }

    /**
     * The "booting" method of the model.
     *
     * @return void
     */
    protected static function boot()
    {
        parent::boot();
        // Activate account on Password Reset.
        Event::listen('Illuminate\Auth\Events\PasswordReset', function ($model) {
            if (is_null($model->user->email_verified_at)) {
                $user = User::find($model->user->id);
                $user->fill(['email_verified_at' => \Carbon\Carbon::now()]);
                $user->save();
            }
        });
    }

    //get last login value by build relation ship with activity log tabe
    public function lastLogin()
    {
        return $this->hasOne('App\ActivityLog')->where('activity', '=', 'login')->orderBy('id', 'desc');
    }

    //get full name
    public function getFullnameAttribute()
    {
        return "{$this->first_name} {$this->last_name}";
    }

    public function getFirmnameAttribute()
    {
        $name = isset($this->firm->name) && !empty($this->firm->name) ? $this->firm->name : null;
        return $name;
    }

    public function getIsStateSelectedAttribute()
    {
        return false;
    }

    public function getIsCompletedProfileAttribute()
    {
        return $this->attributes['is_completed_profile'] == 0 ? false : true;
    }

    public function getCapturenameAttribute()
    {
        $string = preg_replace('/[^a-zA-Z0-9_.]/', '', $this->first_name . ' ' . $this->last_name);
        return str_replace(' ', '_', $string);
    }

    public function getBirthdateAttribute()
    {
        if (isset($this->attributes['birthdate'])) {
            $date = \Carbon\Carbon::parse($this->attributes['birthdate'])->format(config('app.date_format'));
            return $date;
        }
    }

    public function getProfileImageOrgAttribute()
    {
        return $this->attributes['profile_image'] ?? "";
    }

    public function getProfileImageAttribute()
    {
        if (isset($this->attributes['profile_image']) && !empty($this->attributes['profile_image'])) {
            return getImageUrl($this->attributes['profile_image'], 'profile_image');
        } else {
            return asset('assets/admin/images/cms/avatar.png');
        }
    }



    /**
     * Get the user list based on search criteria.
     * @param  \Illuminate\Http\Request  $request
     * @return object App\User
     */
    public function getResult($request)
    {
        // Set default parameter values
        // Curretly it is not in Admin user listing as we are using data table sorting
        // but have kept this for API integration
        $defaultOrder = ($request->get('order_by') == 'name') ? 'first_name' : $request->get('order_by');
        $order_by = !empty($defaultOrder) ? $defaultOrder : 'first_name';
        $order = !empty($request->get('order')) ? $request->get('order') : 'asc';

        // Fetch users list
        $users = new User;

        // Search
        if (!empty($request->get('search'))) {
            $searchStr = $request->get('search');

            $escape = "ESCAPE '|'";
            if (substr_count($searchStr, "|")) {
                $searchStr = str_replace('\\', '\\\\\\', $searchStr);
                $escape = "";
            }
            //$searchStr =  str_replace("|",'\\',$searchStr);
            // added escape for searching backslash issue DLC-140
            $users = $users->where(function ($query) use ($searchStr, $escape) {
                $query
                    ->whereRaw('first_name LIKE ? ' . $escape, '%' . $searchStr . '%')
                    ->orWhereRaw('last_name LIKE ?  ' . $escape, '%' . $searchStr . '%')
                    ->orWhereRaw('email LIKE ?  ' . $escape, '%' . $searchStr . '%')
                    ->orWhereRaw('CONCAT(first_name, " ",last_name) LIKE ?  ' . $escape, '%' . $searchStr . '%');
            });
        }

        // Status
        if ($request->get('status') !== null) {
            $users = $users->where('status', $request->get('status'));
        }

        if ($request->get('bar_id') !== null) {
            $user_ids = BarUserReference::where('bar_id',$request->get('bar_id'))->pluck('user_id');
            $users = $users->whereIn('id', $user_ids);
        }

        if ($request->get('type') !== null) {
            $users = $users->where('user_type_id', $request->get('type'));
        } elseif (!isset($request->group_users)) {
            $users = $users->whereIn('user_type_id', ['1', '2', '3', '4', '5', '6']);
        }
        if (isset($request->not_in_user_type) && $request->not_in_user_type != null) {
            $users = $users->whereNotIn('user_type_id', $request->not_in_user_type);
        }

        if (Auth::user()->user_type_id == 2) {
            $authuser = Auth::user();
            // add firm logic here
            $users = $users->where('firmcode', $authuser->firm->code);
        }

        // Filter for current members based on stripe subscription status
        if (siteconfig('show_only_current_members_app_user') == 1 &&
    $request->get('type') == 3) {
            $users = $users->whereHas('subscriptions', function($query) {
                $query->where(function($q) {
                    $q->where('stripe_subscription_status', 'active');
                });
            })
            ->where('email_verified_at', '!=', null);
        }

        //if per_page not set then all record value use for per_page
        // Curretly it is not in Admin user listing as we are using data table paging
        // but have kept this for API integration
        $per_page = !empty($request->get('per_page')) ? $request->get('per_page') : $users->count();
        if ($order_by == 'first_name') {
            $users = $users->orderBy($order_by, $order)->orderBy('last_name', $order);
        } else {
            $users = $users->orderBy($order_by, $order);
        }

        $users = $users->paginate($per_page);
        return $users;
    }

    /**
     * Send the password reset notification.
     *
     * @param  string  $token
     * @return void
     */
    public function sendPasswordResetNotification($token)
    {
        $this->notify(new MailResetPasswordNotification($token));
    }

    /**
     * Get all active user types
     */
    public function getAssociateType()
    {
        return UserType::select('id', 'title')->where('status', 1)->where('deleted_at', null)->distinct()->get();
    }

    /**
     * Get all active user types
     */
    public function getAssociateAdminRole()
    {
        return UserType::select('id', 'title')->where('status', 1)->where('id', '!=', 3)->where('deleted_at', null)->distinct()->get();
    }

    /**
     * User bars join
     */
    public function barReference()
    {
        return $this->hasMany('App\BarUserReference', 'user_id', 'id')->select('bar_id', 'bar_number');
    }

    public function checkResetLinkExpire($email, $token = null)
    {
        $user = User::where('email', $email)->where('user_type_id', 3)->first();
        if ($user == null) {
            return true;
        }
        return app(\Illuminate\Auth\Passwords\PasswordBroker::class)->tokenExists($user, $token);
    }

    function getProviderAdminList()
    {
        return User::select(DB::raw("CONCAT(first_name, ' ', last_name) AS fullname"), 'id')->where('status', 1)->where('user_type_id', 1)->pluck('id', 'fullname')->flip();
    }

    public function getUserBarActiveIds()
    {
        return BarUserReference::where('user_id', $this->id)->where('is_paid','1')->pluck('bar_id');
    }

    public function getUserBarIds()
    {
        return BarUserReference::where('user_id', $this->id)->whereIn('is_paid',['1','0'])->pluck('bar_id');
    }

    public function getUserBarIdsNeedToPay($id = 0,$ForForcedRenew=0)
    {
        $query = BarUserReference::where('user_id', $this->id);
        if($id != 0){
            $query->where('id',$id);
        }
        if($ForForcedRenew != 1){
            $query->whereIn('is_paid',['2','0']);
        }
        return $query->pluck('bar_id');
    }

    public function getUserBarNeedToPay()
    {
        return BarUserReference::where('user_id', $this->id)->whereIn('is_paid',['2','0'])->get();
    }

    public function getUserBars($bar_id = 0)
    {
        if($bar_id == 0){
            return BarUserReference::where('user_id', $this->id)->get();
        }else{
            return BarUserReference::where('user_id', $this->id)->where('bar_id',$bar_id)->get();
        }

    }

    public function getUserFirmNameAttribute()
    {
        if($this->user_type_id == 3 && $this->appUserFirm){
            return $this->appUserFirm->name??"";
        }else{
            return "";
        }
    }
    public function getUserFirmAddress1Attribute()
    {
        if($this->user_type_id == 3 && $this->appUserFirm){
            return $this->appUserFirm->address1;
        }else{
            return null;
        }
    }
    public function getUserFirmAddress2Attribute()
    {
        if($this->user_type_id == 3 && $this->appUserFirm){
            return $this->appUserFirm->address2;
        }else{
            return null;
        }
    }
    public function getUserFirmCityAttribute()
    {
        if($this->user_type_id == 3 && $this->appUserFirm){
            return $this->appUserFirm->city;
        }else{
            return null;
        }
    }
    public function getUserFirmStateIdAttribute()
    {
        if($this->user_type_id == 3 && $this->appUserFirm){
            return $this->appUserFirm->state_id??0;
        }else{
            return 0;
        }
    }
    public function getUserFirmZipAttribute()
    {
        if($this->user_type_id == 3 && $this->appUserFirm){
            return $this->appUserFirm->zipcode;
        }else{
            return null;
        }
    }

    public function getFullnameWhereId($id)
    {
        $user=User::select('first_name','last_name')->where('id',$id)->first();
        return "{$user->first_name} {$user->last_name}";
    }

    public function getActiveSubscription($bar_id)
    {
        $user_id=$user_id = $this->id;
        $user_bar_subscription= Subscription::where('bar_id',$bar_id)->where('user_id',$user_id)->where('stripe_subscription_status','active')->where('status',1)->first();
        return $user_bar_subscription;
    }

    public function getCreditsSubmittedForBar($user_bar_subscription)
    {
        $user_bar_subscription_credit_submitted = ($user_bar_subscription->hours_needed_per_cycle)-($user_bar_subscription->remaining_hours_needed_per_cycle);
        $creditSubmittedForBar= $user_bar_subscription->remaining_hours_needed_per_cycle == $user_bar_subscription->hours_needed_per_cycle ? 0 : ($user_bar_subscription->remaining_hours_needed_per_cycle == 0 ? $user_bar_subscription->hours_needed_per_cycle : round(($user_bar_subscription_credit_submitted), 2));
        return $creditSubmittedForBar;
    }

    public function getProfile(){

        $user = $this;
        $user = $user->toArray();
        $user['statebars'] = getBarsOfUser($user['id']);

        foreach ($user['statebars'] as $state) {
            $user_bar_subscription= $this->getActiveSubscription($state->bar_id);
            if($user_bar_subscription){
                $pendingSubmissions = UserCredits::where('user_id', $user['id'])
                ->where('bar_id', $state->bar_id)
                ->where('submission_id', 0)
                ->distinct('content_file_id')
                ->count('content_file_id');

                $state->barCreditsSum= $user_bar_subscription->hours_needed_per_cycle ?? 0;
                $state->barCreditsSubmittedSum= $this->getCreditsSubmittedForBar($user_bar_subscription);
                $state->pendingSubmissions = $pendingSubmissions;
            }else{
                $state->barCreditsSum= 0;
                $state->barCreditsSubmittedSum= 0;
                $state->pendingSubmissions = 0;
            }
        }

        unset($user['bar_reference']);
        $user['is_state_selected'] = (count($user['statebars']) > 0) ? true : false;

        $subscription = $this->subscriptions()->get();
        $status = $subscription->isNotEmpty();

        $user['is_payment_done'] = $status;
        unset($user["bar_reference"]);
        unset($user["securitycode"]);
        $user['stripe_publishable_key'] = env('STRIPE_KEY');
        $this->setProfile($user);
        return $user;
    }
}
