<?php

namespace App;

use SoftDeletes;
use App\Model;
use Auth;
class UserType extends Model
{
    use SoftDeletes;

    protected $casts = [
        'permission' => 'array',
    ];

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'title',
        'shortname',
        'redirect_url',
        'sortorder',
        'permission',
        'status',
    ];

    /**
     * Define feilds name which have html tags
     * Set @var notStripTags add DB Table column name which column have html tags.
     *
     * @var array
     */

    public static $notStripTags = ['permission'];

    /**
     * User & User Type Relationship [Many to One]
     * E.g.: Multiple Users are Available with Admin User Role.
     */
    public function user()
    {
        return $this->hasMany('App\User');
    }

    /**
     * Get permission based on User role
     *
     * @param int $id
     * @return array
     */
    public static function getUserTypePermission($role_id)
    {
        return UserType::select('permission')->where('id', $role_id)->first()['permission'];
    }

    /**
     * Get all permission list
     *
     * @return Collection
     */
    public static function getAllUserRolePermission()
    {
        return collect(config('userrolepermissions'))->pluck('permissions')->flatten(1);
    }

    /**
     * check active/inactive role
     *
     *  @param array $param
     * @return boolean
     */
    public static function isRoleActive($user_roles)
    {
        $is_active = false;
        foreach ($user_roles as $role_id) {
            $result = UserType::select('title')->where('id', $role_id)->where('status', '1')->first();
            if (!empty($result)) {
                $is_active = true;
                break;
            }
        }
        return $is_active;
    }



    /**
     * Check user role has permission to access the module
     *
     * @param string $module
     * @param array $param
     * @param string $type key|route
     * @return boolean
     */
    public static function checkHasPermission($module, $param = [], $type = 'key', $user_type = 1)
    {
        if(Auth::user()->user_type_id == 1){
            return true;
        }
        $has_permission = false;
        $all_permissions = self::getAllUserRolePermission();
        $user_role_permissions = self::getUserTypePermission($user_type);
        switch ($type) {
            case "key": {
                    $has_permission = in_array($module, $user_role_permissions);
                    break;
                }
            case "route": {
                    foreach ($all_permissions as $k => $v) {
                        if (isset($v['routes'])) {
                            foreach ($v['routes'] as $route) {
                                $config_param = !empty($route['param']) ? $route['param'] : [];
                                if (((is_array($route['route']) && in_array($module, $route['route'])) || (is_string($route['route']) && $route['route'] == $module)) && (empty($config_param) || (!empty($config_param) && $config_param == $param))) {
                                    $has_permission = in_array($v['key'], $user_role_permissions);
                                }
                            }
                        } else {
                            $config_param = !empty($v['param']) ? $v['param'] : [];
                            if (((is_array($v['route']) && (in_array($module, $v['route']))) || (is_string($v['route']) && $v['route'] == $module)) && (empty($config_param) || (!empty($config_param) && $config_param == $param)) && $user_role_permissions != null) {
                                    $has_permission = in_array($v['key'], $user_role_permissions);
                            }
                        }

                        if ($has_permission) {
                            break;
                        }
                    }
                    break;
                }
        }

        return $has_permission;
    }


    /**
     * The get result based on search criteria.
     * @param  \Illuminate\Http\Request  $request
     * @return object App\Faq
     */
    public function getResult($request)
    {
        // Set default parameter values
        $order_by = !empty($request->get('order_by')) ? $request->get('order_by') : 'shortname';
        $order = !empty($request->get('order')) ? $request->get('order') : 'asc';

        // Fetch faqs list
        $userRole = new UserType();

        // Search
        if (!empty($request->get('search'))) {
            $searchStr = $request->get('search');
            $escape = "ESCAPE '|'";
            if (substr_count($searchStr, "|")) {
                $searchStr = str_replace('\\', '\\\\\\', $searchStr);
                $escape = "";
            }
            //$searchStr = addCslashes($searchStr, '\\');
            // added escape for searching backslash issue DLC-140
            $userRole = $userRole->whereRaw('shortname LIKE ? ' . $escape, '%' . $searchStr . '%');
        }

        // Status
        if ($request->get('status') !== null) {
            $userRole = $userRole->where('status', $request->get('status'));
        }

        // Order By & Pagination
        $userRole = $userRole->orderBy($order_by, $order)->get();

        return $userRole;
    }
}
