
document.addEventListener("DOMContentLoaded", function () {
    
    document.querySelectorAll("audio[id^='song-']").forEach((audio) => {
        const audioId = audio.id.split("-")[1];

        document.querySelectorAll("audio").forEach((audio) => {
            audio.addEventListener("play", function () {
                document.querySelectorAll("audio").forEach((otherAudio) => {
                    if (otherAudio !== audio) {
                        otherAudio.pause();
                        // Optionally reset icon state if you're using play/pause icons
                        let id = otherAudio.id.split('-')[1];
                        let otherCtrlIcon = document.getElementById(`ctrlIcon-${id}`);
                        if (otherCtrlIcon) {
                            otherCtrlIcon.classList.remove("fa-pause");
                            otherCtrlIcon.classList.add("fa-play");
                        }
                    }
                });
            });
        });
        
        const song = document.getElementById(`song-${audioId}`);
        const ctrlIcon = document.getElementById(`ctrlIcon-${audioId}`);
        const playbutton = document.getElementById(`playbutton-${audioId}`);
        const rewindbtn = document.querySelector(`.rewindbtn-${audioId}`);
        const forwardbtn = document.querySelector(`.forwardbtn-${audioId}`);
        const progress = document.getElementById("progress");
        const currentTimeDisplay = document.querySelector(".timer .left");
        const totalTimeDisplay = document.querySelector(".timer .right");
        const loader = document.querySelector(".loader");
        const volumeSlider = document.querySelector(".volume");
        const volumeIcon = document.querySelector(".volume-button i");

        // Safari volume button logic
        let volumeButton = document.querySelector(".volume-button");
        volumeButton.onclick = () => {};
        document.addEventListener("click", (event) => {
            if (!volumeButton.contains(event.target)) {
                volumeButton.classList.remove("active");
            }
        });

        // Volume Control via Web Audio API
        const audioContext = new (window.AudioContext || window.webkitAudioContext)();
        const track = audioContext.createMediaElementSource(song);
        const gainNode = audioContext.createGain();
        track.connect(gainNode);
        gainNode.connect(audioContext.destination);

        function setVolume(value) {
            const vol = value / 100;
            gainNode.gain.value = vol;
            updateVolumeIcon(vol);
        }

        function updateVolumeIcon(vol) {
            if (vol === 0) {
                volumeIcon.classList.remove("fa-volume-up", "fa-volume-down");
                volumeIcon.classList.add("fa-volume-off");
            } else if (vol < 0.5) {
                volumeIcon.classList.remove("fa-volume-up", "fa-volume-off");
                volumeIcon.classList.add("fa-volume-down");
            } else {
                volumeIcon.classList.remove("fa-volume-down", "fa-volume-off");
                volumeIcon.classList.add("fa-volume-up");
            }
        }

        volumeSlider.addEventListener("input", () => setVolume(volumeSlider.value));
        setVolume(volumeSlider.value);

        // Resume audio context on interaction (Safari requirement)
        function unlockAudio() {
            if (audioContext.state === "suspended") {
                audioContext.resume();
            }
            document.removeEventListener("click", unlockAudio);
        }
        document.addEventListener("click", unlockAudio);

        function formatTime(seconds) {
            const mins = Math.floor(seconds / 60);
            const secs = Math.floor(seconds % 60);
            return `${mins}:${secs < 10 ? "0" + secs : secs}`;
        }

        song.onloadedmetadata = () => {
            if (progress) {
                progress.max = song.duration;
                progress.value = song.currentTime;
            }
            if (totalTimeDisplay) {
                totalTimeDisplay.textContent = formatTime(song.duration);
            }
        };

        song.addEventListener("timeupdate", () => {
            if (progress) {
                progress.value = song.currentTime;
            }
            if (currentTimeDisplay) {
                currentTimeDisplay.textContent = formatTime(song.currentTime);
            }
        });

        if (progress) {
            progress.onchange = () => {
                song.currentTime = progress.value;
                song.play();
                ctrlIcon.classList.add("fa-pause");
                ctrlIcon.classList.remove("fa-play");
            };
        }

        // Play/Pause Button
        playbutton.addEventListener("click", () => {
            if (ctrlIcon.classList.contains("fa-pause")) {
                song.pause();
                ctrlIcon.classList.remove("fa-pause");
                ctrlIcon.classList.add("fa-play");
            } else {
                song.play();
                ctrlIcon.classList.add("fa-pause");
                ctrlIcon.classList.remove("fa-play");
            }
        });

        rewindbtn.addEventListener("click", () => {
            song.currentTime = Math.max(0, song.currentTime - 15);
        });

        forwardbtn.addEventListener("click", () => {
            song.currentTime = Math.min(song.duration, song.currentTime + 15);
        });

        function disableButtonsWhileLoading() {
            document.querySelector(".loader-bg")?.classList.add("loading");
            playbutton.style.pointerEvents = "none";
            rewindbtn.style.pointerEvents = "none";
            forwardbtn.style.pointerEvents = "none";
        }

        function enableButtonsWhenLoaded() {
            document.querySelector(".loader-bg")?.classList.remove("loading");
            playbutton.style.pointerEvents = "auto";
            rewindbtn.style.pointerEvents = "auto";
            forwardbtn.style.pointerEvents = "auto";
        }

        song.addEventListener("waiting", disableButtonsWhileLoading);
        song.addEventListener("canplay", enableButtonsWhenLoaded);
        song.addEventListener("playing", enableButtonsWhenLoaded);
        song.addEventListener("error", () => {
            enableButtonsWhenLoaded();
            setTimeout(disableButtonsWhileLoading, 3000);
        });

        // Disable progress interaction (as per requirement)
        if (progress) {
            progress.style.pointerEvents = "none";
        }
    });
});

