<?php

namespace Tests\Feature\Controllers\Admin;

use App\User;
use App\Category;
use App\Blog;
use App\CategoryReference;
use Tests\TestCase;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Foundation\Testing\RefreshDatabase;

class BlogControllerTest extends TestCase
{
    use RefreshDatabase, WithFaker;
    protected $indexRoute='blog.index';
    protected $storeRoute='blog.store';
    protected $bulkRoute='blog.bulkaction';

    protected function setUp(): void
    {
        parent::setUp();

        $this->user = factory(User::class)->create();
    }

    /** @test */
    public function testIndex()
    {
        $this->actingAs($this->user)
            ->get(route($this->indexRoute))
            ->assertOk()
            ->assertViewIs('admin.modules.blog.index')
            ->assertViewHas(['blogs', 'blog_categories']);
    }

    /** @test */
    public function testCreate()
    {
        $this->actingAs($this->user)
            ->get(route('blog.create'))
            ->assertOk()
            ->assertViewIs('admin.modules.blog.addedit')
            ->assertViewHas('blog');
    }

    /** @test */
    public function testValidations()
    {
        $blog = factory(Blog::class)->raw([
            'title' => '',
            'publish_date' => '',
            'description' => '',
            'main_image' => '',
            'main_image_alt' => '',
            'meta_title' => '',
            'meta_desc' => '',
            'status' => '',
        ]);

        $this->actingAs($this->user)
            ->post(route($this->storeRoute), $blog)
            ->assertSessionHasErrors([
                'title' => 'The title field is required.',
                'publish_date' => 'The publish date field is required.',
                'description' => 'The description field is required.',
                'main_image' => 'The main image field is required.',
                'main_image_alt' => 'The main image alt field is required.',
                'meta_title' => 'The meta title field is required.',
                'meta_desc' => 'The meta description field is required.',
                'status' => 'The status field is required.'
            ]);
    }

    /** @test */
    public function testDateValidations()
    {
        $this->actingAs($this->user)
            ->post(route($this->storeRoute), factory(Blog::class)->raw([
                'publish_date' => $this->faker->word,
            ]))
            ->assertSessionHasErrors([
                'publish_date' => 'The publish date is not a valid date.',
            ]);
    }

    /** @test */
    public function testStore()
    {
        $category = factory(Category::class, 'db_record')->create();

        $blog = factory(Blog::class)->raw([
            'parent_category' => $category->id,
            'btnsave' => 'save'
        ]);

        $this->actingAs($this->user)
            ->post(route($this->storeRoute), $blog)
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('blog.create_success', ['title' => $blog['title']]));
    }

    /** @test */
    public function testEdit()
    {
        $category = factory(Category::class, 'db_record')->create();

        $blog = factory(Blog::class, 'db_record')->create();

        CategoryReference::create([
            'ref_model_name' => 'blog',
            'ref_id' => $blog->id,
            'category_id' => $category->id,
        ]);

        $this->actingAs($this->user)
            ->get(route('blog.edit', $blog->id))
            ->assertOk()
            ->assertViewIs('admin.modules.blog.addedit')
            ->assertViewHas('blog');
    }

    /** @test */
    public function testUpdate()
    {
        $categories = factory(Category::class, 'db_record', 2)->create();

        $blog = factory(Blog::class, 'db_record')->create();

        CategoryReference::create([
            'ref_model_name' => 'blog',
            'ref_id' => $blog->id,
            'category_id' => $categories->first()->id,
        ]);

        $data = factory(Blog::class)->raw([
            'parent_category' => $categories->get(1)->id,
            'btnsave' => 'save'
        ]);

        $this->actingAs($this->user)
            ->put(route('blog.update', ['blog' => $blog->id]), $data)
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('blog.update_success', ['title' => $data['title']]));
    }

    /** @test */
    public function testUpdateAndContinue()
    {
        $categories = factory(Category::class, 'db_record', 2)->create();

        $blog = factory(Blog::class, 'db_record')->create();

        CategoryReference::create([
            'ref_model_name' => 'blog',
            'ref_id' => $blog->id,
            'category_id' => $categories->first()->id,
        ]);

        $data = factory(Blog::class)->raw([
            'parent_category' => $categories->get(1)->id,
            'btnsave' => 'savecontinue'
        ]);

        $this->actingAs($this->user)
            ->put(route('blog.update', ['blog' => $blog->id]), $data)
            ->assertStatus(302)
            ->assertRedirect(route('blog.edit', ['blog' => $blog->id]))
            ->assertSessionHas('success', __('blog.update_success', ['title' => $data['title']]));
    }

    /** @test */
    public function testChangeStatusToInactive()
    {
        $blog = factory(Blog::class)->states('active')->create();

        $this->actingAs($this->user)
            ->from(route($this->indexRoute))
            ->get(route('blog.changestatus', ['id' => $blog->id]))
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('blog.record_inactive_success'));
    }

    /** @test */
    public function testChangeStatusToActive()
    {
        $blog = factory(Blog::class)->states('inactive')->create();

        $this->actingAs($this->user)
            ->from(route($this->indexRoute))
            ->get(route('blog.changestatus', ['id' => $blog->id]))
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('blog.record_active_success'));
    }

    /** @test */
    public function testBulkActionActive()
    {
        $blogs = factory(Blog::class, 2)->create();

        $this->actingAs($this->user)
            ->from(route($this->indexRoute))
            ->post(route($this->bulkRoute, ['bulk-action' => 'active', 'id' => $blogs->pluck('id')->all()]))
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('blog.active_success'));
    }

    /** @test */
    public function testBulkActionInactive()
    {
        $blogs = factory(Blog::class, 2)->create();

        $this->actingAs($this->user)
            ->from(route($this->indexRoute))
            ->post(route($this->bulkRoute, ['bulk-action' => 'inactive', 'id' => $blogs->pluck('id')->all()]))
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('blog.inactive_success'));
    }

    /** @test */
    public function testBulkActionDelete()
    {
        $blogs = factory(Blog::class, 2)->create();

        $this->actingAs($this->user)
            ->from(route($this->indexRoute))
            ->post(route($this->bulkRoute, ['bulk-action' => 'delete', 'id' => $blogs->pluck('id')->all()]))
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('blog.delete_success'));
    }
}
