<?php

namespace Tests\Feature\Controllers\Admin;

use App\User;
use App\Category;
use App\Blog;
use App\CategoryReference;
use Tests\TestCase;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Foundation\Testing\RefreshDatabase;

class CategoryControllerTest extends TestCase
{
    use RefreshDatabase, WithFaker;
    protected $indexRoute='category.index';
    protected $bulkRoute='category.bulkaction';

    protected function setUp(): void
    {
        parent::setUp();

        $this->user = factory(User::class)->create();
    }

    /** @test */
    public function testIndex()
    {
        $this->actingAs($this->user)
            ->get(route($this->indexRoute))
            ->assertOk()
            ->assertViewIs('admin.modules.category.index')
            ->assertViewHas('categories');
    }

    /** @test */
    public function testCreate()
    {
        $this->actingAs($this->user)
            ->get(route('category.create'))
            ->assertOk()
            ->assertViewIs('admin.modules.category.addedit')
            ->assertViewHas('category');
    }

    /** @test */
    public function testValidations()
    {
        $category = factory(Category::class)->raw([
            'title' => '',
            'status' => '',
        ]);

        $this->actingAs($this->user)
            ->post(route('category.store'), $category)
            ->assertSessionHasErrors([
                'title' => 'The title field is required.',
                'status' => 'The status field is required.'
            ]);
    }

    /** @test */
    public function testStore()
    {
        $category = factory(Category::class)->raw([
            'btnsave' => 'save'
        ]);

        $this->actingAs($this->user)
            ->post(route('category.store'), $category)
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('category.create_success', ['title' => $category['title']]));
    }

    /** @test */
    public function testEdit()
    {
        $category = factory(Category::class)->create();

        $this->actingAs($this->user)
            ->get(route('category.edit', $category->id))
            ->assertOk()
            ->assertViewIs('admin.modules.category.addedit')
            ->assertViewHas('category');
    }

    /** @test */
    public function testUpdate()
    {
        $category = factory(Category::class)->create();

        $data = factory(Category::class)->raw([
            'btnsave' => 'save'
        ]);

        $this->actingAs($this->user)
            ->put(route('category.update', ['category' => $category->id]), $data)
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('category.update_success', ['title' => $data['title']]));
    }

    /** @test */
    public function testUpdateAndContinue()
    {
        $category = factory(Category::class)->create();

        $data = factory(Category::class)->raw([
            'btnsave' => 'savecontinue'
        ]);

        $this->actingAs($this->user)
            ->put(route('category.update', ['category' => $category->id]), $data)
            ->assertStatus(302)
            ->assertRedirect(route('category.edit', ['category' => $category->id]))
            ->assertSessionHas('success', __('category.update_success', ['title' => $data['title']]));
    }

    /** @test */
    public function testChangeStatusToInactive()
    {
        $category = factory(Category::class)->states('active')->create();

        $this->actingAs($this->user)
            ->from(route($this->indexRoute))
            ->get(route('category.changestatus', ['category' => $category->id]))
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('category.record_inactive_success'));
    }

    /** @test */
    public function testChangeStatusToActive()
    {
        $category = factory(Category::class)->states('inactive')->create();

        $this->actingAs($this->user)
            ->from(route($this->indexRoute))
            ->get(route('category.changestatus', ['category' => $category->id]))
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('category.record_active_success'));
    }

    /** @test */
    public function testBulkActionActive()
    {
        $categories = factory(Category::class, 2)->create();

        $this->actingAs($this->user)
            ->from(route($this->indexRoute))
            ->post(route($this->bulkRoute, ['bulk-action' => 'active', 'id' => $categories->pluck('id')->all()]))
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('category.active_success'));
    }

    /** @test */
    public function testBulkActionInactive()
    {
        $categories = factory(Category::class, 2)->create();

        $this->actingAs($this->user)
            ->from(route($this->indexRoute))
            ->post(route($this->bulkRoute, ['bulk-action' => 'inactive', 'id' => $categories->pluck('id')->all()]))
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('category.inactive_success'));
    }

    /** @test */
    public function testBulkActionDelete()
    {
        $categories = factory(Category::class, 2)->create();

        $this->actingAs($this->user)
            ->from(route($this->indexRoute))
            ->post(route($this->bulkRoute, ['bulk-action' => 'delete', 'id' => $categories->pluck('id')->all()]))
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('category.delete_success'));
    }

    /** @test */
    public function testBulkActionDeleteBlogReference()
    {
        $categories = factory(Category::class, 2)->create();

        $blog = factory(Blog::class, 'db_record')->create();

        CategoryReference::create([
            'ref_model_name' => 'blog',
            'ref_id' => $blog->id,
            'category_id' => $categories->pluck('id')->first(),
        ]);

        $this->actingAs($this->user)
            ->from(route($this->indexRoute))
            ->post(route($this->bulkRoute, ['bulk-action' => 'delete', 'id' => $categories->pluck('id')->all()]))
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('category.delete_success'));
    }

    /** @test */
    public function testDependantData()
    {
        $categories = factory(Category::class, 2)->create();

        $blog = factory(Blog::class, 'db_record')->create();

        CategoryReference::create([
            'ref_model_name' => 'blog',
            'ref_id' => $blog->id,
            'category_id' => $categories->pluck('id')->first(),
        ]);

        $this->actingAs($this->user)
            ->post(route('category.checkdependantdata', ['id' => $categories->pluck('id')->all()]))
            ->assertOk()
            ->assertJson([
                'success' => false,
                'categories' => [$categories->pluck('title')->first()],
            ]);
    }

    /** @test */
    public function testDependantDataWithoutReference()
    {
        $categories = factory(Category::class, 2)->create();

        $this->actingAs($this->user)
            ->post(route('category.checkdependantdata', ['id' => $categories->pluck('id')->all()]))
            ->assertOk()
            ->assertJson([
                'success' => true,
                'categories' => [],
            ]);
    }
}
