<?php

namespace Tests\Feature\Controllers\Admin;

use App\Event;
use App\User;
use Tests\TestCase;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Foundation\Testing\RefreshDatabase;

class EventControllerTest extends TestCase
{
    use  WithFaker;

    protected $indexRoute='event.index';
    protected $storeRoute='event.store';
    protected $bulkRoute='event.bulkaction';

    protected function setUp(): void
    {
        parent::setUp();
        $this->user = factory(User::class)->create();
    }

    /** @test for listing events */
    public function testIndex()
    {
         $this->actingAs($this->user)
            ->get(route($this->indexRoute))
            ->assertOk()
            ->assertViewIs('admin.modules.event.index')
            ->assertViewHas('events');
    }

    /** @test for creating event*/
    public function testCreate()
    {
        $this->actingAs($this->user)
            ->get(route('event.create'))
            ->assertOk()
            ->assertViewIs('admin.modules.event.addedit')
            ->assertViewHas('event');
    }

    /** @test for all required validation*/
    public function testRequiredValidation()
    {
        $events = array('title'=>'', 'start_date' => '', 'end_date' => '', 'event_time' => '');
        $this->actingAs($this->user)
            ->post(route($this->storeRoute), $events)
            ->assertStatus(302)
            ->assertSessionHasErrors(['title' => 'The title field is required.',
                                      'start_date' => 'The start date field is required.',
                                      'end_date' => 'The end date field is required.',
                                      'event_time' => 'The event time field is required.']);
    }

    /** @test for url validation */
    public function testUrlValidation()
    {
        collect(['www.mcglynn.biz', 'https://www.mcglynn,biz', 'mcglynn.biz'])->each(function ($invalidUrl) {
            $this->actingAs($this->user)
                ->post(
                    route($this->storeRoute),
                    factory(Event::class)->raw([
                        'url' => $invalidUrl,
                    ])
                )->assertSessionHasErrors([
                    'url' => 'The url format is invalid.'
                ]);
        });
    }

    /** @test for invalid date validation */
    public function testDateValidation()
    {
        collect(['0901202a', '091202c'])->each(function ($invalidDate) {
            $this->actingAs($this->user)
                ->post(
                    route($this->storeRoute),
                    factory(Event::class)->raw([
                        'start_date' => $invalidDate,
                        'end_date' => $invalidDate,
                    ])
                )->assertSessionHasErrors([
                    'start_date' => 'The start date is not a valid date.',
                    'end_date' => 'The end date is not a valid date.'
                ]);
        });
    }

    /** @test for before_or_equal,after_or_equal validation */
    public function testDateBeforeAfterOrEqualValidation()
    {
        $this->actingAs($this->user)
        ->post(
            route($this->storeRoute),
            factory(Event::class)->raw([
                'start_date' => '1/9/2020',
                'end_date' => '1/7/2020',
            ])
        )->assertSessionHasErrors([
            'start_date' => 'The start date must be a date before or equal to end date.',
            'end_date' => 'The end date must be a date after or equal to start date.'
        ]);
    }

    /** @test for store Event */
    public function testStore()
    {
        $event = factory(Event::class)->raw([
            'btnsave' => 'save'
        ]);

         $this->actingAs($this->user)
            ->post(route($this->storeRoute), $event)
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('event.create_success', ['title' => $event['title']]));
    }

    /** @test for edit Event */
    public function testEdit()
    {
        $event = factory(Event::class)->create();

         $this->actingAs($this->user)
            ->get(route('event.edit', $event->id))
            ->assertOk()
            ->assertViewIs('admin.modules.event.addedit')
            ->assertViewHasAll(['event']);
    }

    /** @test for updating event data in DB */
    public function testUpdate()
    {
        $event = factory(Event::class)->create();

        $data = factory(Event::class)->raw([
            'btnsave' => 'save'
        ]);

         $this->actingAs($this->user)
            ->put(route('event.update', ['event' => $event->id]), $data)
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('event.update_success', ['title' => $data['title']]));
    }

    /** @test for update and continue button*/
    public function testUpdateAndContinue()
    {
        $event = factory(Event::class)->create();

        $data = factory(Event::class)->raw([
            'btnsave' => 'savecontinue'
        ]);

         $this->actingAs($this->user)
            ->put(route('event.update', ['event' => $event->id]), $data)
            ->assertStatus(302)
            ->assertRedirect(route('event.edit', $event->id))
            ->assertSessionHas('success', __('event.update_success', ['title' => $data['title']]));
    }

    /** @test for bulk action active */
    public function testBulkActionActive()
    {
        $events = factory(Event::class, 2)->states('inactive')->create();

         $this->actingAs($this->user)
            ->from(route($this->indexRoute))
            ->post(route($this->bulkRoute, ['bulk-action' => 'active', 'id' => $events->pluck('id')->all()]))
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('event.active_success'));
    }

    /** @test for bulk action inactive */
    public function testBulkActionInactive()
    {
        $events = factory(Event::class, 2)->create();

         $this->actingAs($this->user)
            ->from(route($this->indexRoute))
            ->post(route($this->bulkRoute, ['bulk-action' => 'inactive', 'id' => $events->pluck('id')->all()]))
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('event.inactive_success'));
    }

    /** @test for bulk action delete */
    public function testBulkActionDelete()
    {
        $events = factory(Event::class, 2)->create();

         $this->actingAs($this->user)
            ->from(route($this->indexRoute))
            ->post(route($this->bulkRoute, ['bulk-action' => 'delete', 'id' => $events->pluck('id')->all()]))
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('event.delete_success'));
    }

     /** @test change single status inactive */
     public function testChangeStatusToInactive()
     {
         $event = factory(Event::class)->create();

          $this->actingAs($this->user)
             ->from(route($this->indexRoute))
             ->get(route('event.changestatus', ['id' => $event->id]))
             ->assertStatus(302)
             ->assertRedirect(route($this->indexRoute))
             ->assertSessionHas('success', __('event.record_inactive_success'));
     }

     /** @test change single status active */
    public function testChangeStatusToActive()
    {
        $event = factory(Event::class)->states('inactive')->create();

         $this->actingAs($this->user)
            ->from(route($this->indexRoute))
            ->get(route('event.changestatus', ['id' => $event->id]))
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('event.record_active_success'));
    }

}
