<?php

namespace Tests\Feature\Controllers\Admin;

use App\SiteConfig;
use App\User;
use Tests\TestCase;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Foundation\Testing\RefreshDatabase;

class SiteConfigControllerTest extends TestCase
{
    use RefreshDatabase, WithFaker;
    protected $saveSiteconfig = 'site-config.save';

    protected function setUp(): void
    {
        parent::setUp();

        $this->user = factory(User::class)->create();

        $this->modules = ['global', 'team', 'testimonial', 'blog', 'slider', 'contact'];
    }

    /** @test */
    public function testIndex()
    {
        foreach ($this->modules as $module) {
            $this->actingAs($this->user)
                ->get(route('site-config', $module))
                ->assertOk()
                ->assertViewIs('admin.modules.siteconfig.index')
                ->assertViewHas('module');
        }
    }

    /** @test */
    public function testStore()
    {
        foreach ($this->modules as $module) {

            $data = factory(SiteConfig::class, $module)->raw();

            $this->actingAs($this->user)
                ->from(route('site-config', $module))
                ->post(route($this->saveSiteconfig, ['module' => $module]), $data)
                ->assertStatus(302)
                ->assertRedirect(route('site-config', $module))
                ->assertSessionHas('success', __('siteconfig.success'));
        }
    }

    /** @test */
    public function testRequiredValidation()
    {
        $this->modules = ['global', 'team', 'blog', 'contact'];
        foreach ($this->modules as $module) {

            $data = factory(SiteConfig::class, $module)->raw();

            $fieldsToValidate = factory(SiteConfig::class, $module . '-required-fields-and-errors')->raw();

            foreach ($fieldsToValidate as $field => $message) {
                $data[$field] = '';
                $this->actingAs($this->user)
                    ->post(route($this->saveSiteconfig, ['module' => $module]), $data)
                    ->assertSessionHasErrors([$field => $message]);
            }
        }
    }

    /** @test */
    public function testNumericValidation()
    {
        collect([$this->faker->word, '!@#$%'])->each(function ($invalidNumeric) {
            $this->actingAs($this->user)
                ->post(
                    route($this->saveSiteconfig, ['module' => 'global']),
                    factory(SiteConfig::class, 'global')->raw([
                        'company_zipcode' => $invalidNumeric,
                        'login_attempt' => $invalidNumeric,
                        'minimum_password_length' => $invalidNumeric,
                        'user_block_time' => $invalidNumeric,
                    ])
                )
                ->assertSessionHasErrors([
                    'company_zipcode' => 'The company zipcode must be a number.',
                    'login_attempt' => 'The login attempt must be a number.',
                    'minimum_password_length' => 'The minimum password length must be a number.',
                    'user_block_time' => 'The user block time must be a number.',
                ]);
        });
    }

    /** @test */
    public function testCompanyZipcodeDigitsBetweenValidation()
    {
        collect([$this->faker->randomNumber(4), $this->faker->randomNumber(7), 5.5, -5])->each(function ($invalidLengthNumber) {
            $this->actingAs($this->user)
                ->post(
                    route($this->saveSiteconfig, ['module' => 'global']),
                    factory(SiteConfig::class, 'global')->raw([
                        'company_zipcode' => $invalidLengthNumber,
                    ])
                )
                ->assertSessionHasErrors([
                    'company_zipcode' => 'The company zipcode must be between 5 and 6 digits.',
                ]);
        });
    }

    /** @test */
    public function testEmailValidation()
    {
        collect(['you@example,com', 'bad_user.org', 'example@bad+user.com'])->each(function ($invalidEmail) {
            $this->actingAs($this->user)
                ->post(
                    route($this->saveSiteconfig, ['module' => 'global']),
                    factory(SiteConfig::class, 'global')->raw([
                        'company_email' => $invalidEmail,
                        'from_email' => $invalidEmail,
                    ])
                )->assertSessionHasErrors([
                    'company_email' => 'The company email must be a valid email address.',
                    'from_email' => 'The from email must be a valid email address.',
                ]);
        });
    }

    /** @test */
    public function testGreaterThanValidation()
    {
        collect([0, -1])->each(function ($lessThanOneValue) {
            $this->actingAs($this->user)
                ->post(
                    route($this->saveSiteconfig, ['module' => 'global']),
                    factory(SiteConfig::class, 'global')->raw([
                        'login_attempt' => $lessThanOneValue,
                        'minimum_password_length' => $lessThanOneValue,
                        'user_block_time' => $lessThanOneValue,
                    ])
                )->assertSessionHasErrors([
                    'login_attempt' => 'The login attempt must be greater than or equal 1.',
                    'minimum_password_length' => 'The minimum password length must be greater than or equal 5.',
                    'user_block_time' => 'The user block time must be greater than or equal 1.',
                ]);
        });
    }

    /** @test */
    public function testGreaterThanValidationOfBlog()
    {
        collect([0, -1])->each(function ($lessThanOneValue) {
            $this->actingAs($this->user)
                ->post(
                    route($this->saveSiteconfig, ['module' => 'blog']),
                    factory(SiteConfig::class, 'blog')->raw([
                        'field_blog_per_page_display' => $lessThanOneValue,
                    ])
                )->assertSessionHasErrors([
                    'field_blog_per_page_display' => 'The blog per page display must be greater than or equal 1.',
                ]);
        });
    }

    /** @test */
    public function testLessThanValidation()
    {
        $this->actingAs($this->user)
            ->post(
                route($this->saveSiteconfig, ['module' => 'global']),
                factory(SiteConfig::class, 'global')->raw([
                    'minimum_password_length' => $this->faker->numberBetween(12, 255),
                ])
            )->assertSessionHasErrors([
                'minimum_password_length' => 'The minimum password length must be less than or equal 12.',
            ]);
    }

    /** @test */
    public function testUrlValidation()
    {
        collect(['www.mcglynn.biz', 'https://www.mcglynn,biz', 'mcglynn.biz'])->each(function ($invalidUrl) {
            $this->actingAs($this->user)
                ->post(
                    route($this->saveSiteconfig, ['module' => 'global']),
                    factory(SiteConfig::class, 'global')->raw([
                        'facebook_link' => $invalidUrl,
                        'twitter_link' => $invalidUrl,
                        'linkedin_link' => $invalidUrl,
                        'instagram_link' => $invalidUrl,
                    ])
                )->assertSessionHasErrors([
                    'facebook_link' => 'The facebook link format is invalid.',
                    'twitter_link' => 'The twitter link format is invalid.',
                    'linkedin_link' => 'The linkedin link format is invalid.',
                    'instagram_link' => 'The linkedin link format is invalid.',
                ]);
        });
    }

    /** @test */
    public function testAdminEmailArrayValidation()
    {
        collect(['you@example.com,me@example', 'you@example.comtest@example.com', 'example1@bad+user.com'])->each(function ($invalidEmail) {
            $this->actingAs($this->user)
                ->post(
                    route($this->saveSiteconfig, ['module' => 'global']),
                    factory(SiteConfig::class, 'global')->raw([
                        'admin_email' => $invalidEmail,
                    ])
                )->assertSessionHasErrors([
                    'admin_email' => 'The admin email must be a valid email address. Email does not contain space after comma.',
                ]);
        });
    }

    /** @test */
    public function testContactEmailArrayValidation()
    {
        collect(['you@example.com,me@example', 'you@example.comtest@example.com', 'example@bad+user.com'])->each(function ($invalidEmail) {
            $this->actingAs($this->user)
                ->post(
                    route($this->saveSiteconfig, ['module' => 'contact']),
                    factory(SiteConfig::class, 'contact')->raw([
                        'contact_email' => $invalidEmail,
                    ])
                )->assertSessionHasErrors([
                    'contact_email' => 'The contact email must be a valid email address. Email does not contain space after comma.',
                ]);
        });
    }

    /** @test */
    public function testIntegerValidation()
    {
        $this->actingAs($this->user)
            ->post(
                route($this->saveSiteconfig, ['module' => 'blog']),
                factory(SiteConfig::class, 'blog')->raw([
                    'field_blog_per_page_display' => $this->faker->randomFloat(),
                ])
            )->assertSessionHasErrors([
                'field_blog_per_page_display' => 'The blog per page display must be an integer.',
            ]);
    }
}
