<?php

namespace Tests\Feature\Controllers\Admin;

use App\User;
use Tests\TestCase;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Foundation\Testing\RefreshDatabase;

class UserControllerTest extends TestCase
{
    use RefreshDatabase, WithFaker;

    protected $indexRoute='user.index';
    protected $storeRoute='user.store';
    protected $editRoute='user.edit';
    protected $updateRoute='user.update';
    protected $changeRoute='user.changestatus';
    protected $bulkRoute='user.bulkaction';

    protected function setUp(): void
    {
        parent::setUp();

        $this->user = factory(User::class)->create();
    }

    /** @test */
    public function testIndex()
    {
         $this->actingAs($this->user)
            ->get(route($this->indexRoute))
            ->assertOk()
            ->assertViewIs('admin.modules.user.index')
            ->assertViewHas('users');
    }

    /** @test */
    public function testCreate()
    {
         $this->actingAs($this->user)
            ->get(route('user.create'))
            ->assertOk()
            ->assertViewIs('admin.modules.user.addedit')
            ->assertViewHas('user');
    }

    /** @test */
    public function testFirstNameRequiredValidation()
    {
        $user = factory(User::class)->raw([
            'first_name' => '',
        ]);

         $this->actingAs($this->user)
            ->post(route($this->storeRoute), $user)
            ->assertSessionHasErrors(['first_name' => 'The first name field is required.']);
    }

    /** @test */
    public function testEmailRequiredValidation()
    {
        $user = factory(User::class)->raw([
            'email' => '',
        ]);

         $this->actingAs($this->user)
            ->post(route($this->storeRoute), $user)
            ->assertSessionHasErrors(['email' => 'The email field is required.']);
    }

    /** @test */
    public function testEmailValidation()
    {
        collect(['you@example,com', 'bad_user.org', 'example@bad+user.com'])->each(function ($invalidEmail) {
            $this->actingAs($this->user)
                ->post(
                    route($this->storeRoute),
                    factory(User::class)->raw([
                        'email' => $invalidEmail,
                    ])
                )->assertSessionHasErrors([
                    'email' => 'The email must be a valid email address.'
                ]);
        });
    }

    /** @test */
    public function testEmailUniqueValidation()
    {
        $user = factory(User::class)->raw([
            'email' => $this->user->email,
        ]);

         $this->actingAs($this->user)
            ->post(route($this->storeRoute), $user)
            ->assertSessionHasErrors(['email' => 'The email has already been taken.']);
    }

    /** @test */
    public function testStore()
    {
        $user = factory(User::class)->raw([
            'btnsave' => 'save'
        ]);

         $this->actingAs($this->user)
            ->post(route($this->storeRoute), $user)
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('user.create_success', ['first_name' => $user['first_name'], 'last_name' => $user['last_name']]));
    }

    /** @test */
    public function testRestore()
    {
        $deletedUser = factory(User::class)->states('deleted')->create();

        $user = factory(User::class)->raw([
            'email' => $deletedUser->email,
            'btnsave' => 'save',
        ]);

         $this->actingAs($this->user)
            ->post(route($this->storeRoute), $user)
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('user.restored_success', ['first_name' => $user['first_name'], 'last_name' => $user['last_name']]));
    }

    /** @test */
    public function testEdit()
    {
         $this->actingAs($this->user)
            ->get(route($this->editRoute, $this->user->id))
            ->assertOk()
            ->assertViewIs('admin.modules.user.addedit')
            ->assertViewHas('user');
    }

    /** @test */
    public function testUpdate()
    {
        $data = factory(User::class)->raw([
            'btnsave' => 'save'
        ]);

         $this->actingAs($this->user)
            ->put(route($this->updateRoute, ['user' => $this->user->id]), $data)
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('user.update_success', ['first_name' => $data['first_name'], 'last_name' => $data['last_name']]));
    }

    /** @test */
    public function testUpdateExceptStatus()
    {
        $nonVerifiedUser = factory(User::class)->states('not_verified')->create();

        $data = factory(User::class)->raw([
            'btnsave' => 'save'
        ]);

         $this->actingAs($this->user)
            ->put(route($this->updateRoute, ['user' => $nonVerifiedUser->id]), $data)
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('user.update_status_not_valid', ['first_name' => $data['first_name'], 'last_name' => $data['last_name']]));
    }

    /** @test */
    public function testUpdateAndContinue()
    {
        $data = factory(User::class)->raw([
            'btnsave' => 'savecontinue'
        ]);

         $this->actingAs($this->user)
            ->from(route($this->editRoute, ['user' => $this->user->id]))
            ->put(route($this->updateRoute, ['user' => $this->user->id]), $data)
            ->assertStatus(302)
            ->assertRedirect(route($this->editRoute, ['user' => $this->user->id]))
            ->assertSessionHas('success', __('user.update_success', ['first_name' => $data['first_name'], 'last_name' => $data['last_name']]));
    }

    /** @test */
    public function testResetPassword()
    {
        $user = factory(User::class)->create();

         $this->actingAs($this->user)
            ->from(route($this->indexRoute))
            ->get(route('user.reset-password', ['email' => $user->email,'usertypeid' =>  $user->user_type_id]))
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('user.resetpasswordlink', ['email' => $user->email]));
    }

    /** @test */
    public function testAccountActivation()
    {
        $nonVerifiedUser = factory(User::class)->states('not_verified')->create();

         $this->actingAs($this->user)
            ->from(route($this->indexRoute))
            ->get(route('user.reset-password', ['email' => $nonVerifiedUser->email,'usertypeid' =>  $nonVerifiedUser->user_type_id]))
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('user.activationlink_success', ['email' => $nonVerifiedUser->email]));
    }

    /** @test */
    public function testChangeStatusToInactive()
    {
         $this->actingAs($this->user)
            ->from(route($this->indexRoute))
            ->get(route($this->changeRoute, ['id' => $this->user->id]))
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('user.account_inactivated_success'));
    }

    /** @test */
    public function testChangeStatusToActive()
    {
        $user = factory(User::class)->states('inactivated')->create();

         $this->actingAs($this->user)
            ->from(route($this->indexRoute))
            ->get(route($this->changeRoute, ['id' => $user->id]))
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('user.account_activated_success'));
    }

    /** @test */
    public function testChangeStatusToActiveofNotVerifiedUser()
    {
        $user = factory(User::class)->states('not_verified')->create();

         $this->actingAs($this->user)
            ->from(route($this->indexRoute))
            ->get(route($this->changeRoute, ['id' => $user->id]))
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('error', __('user.account_not_validated'));
    }

    /** @test */
    public function testBulkActionActive()
    {
        $users = factory(User::class, 2)->states('inactivated')->create();

         $this->actingAs($this->user)
            ->from(route($this->indexRoute))
            ->post(route($this->bulkRoute, ['bulk-action' => 'active', 'id' => $users->pluck('id')->all()]))
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('user.active_success'));
    }

    /** @test */
    public function testBulkActionActiveofNotVerifiedUsers()
    {
        $users = factory(User::class, 2)->states('not_verified')->create();

         $this->actingAs($this->user)
            ->from(route($this->indexRoute))
            ->post(route($this->bulkRoute, ['bulk-action' => 'active', 'id' => $users->pluck('id')->all()]))
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('user.active_success_validate'));
    }

    /** @test */
    public function testBulkActionInactive()
    {
        $users = factory(User::class, 2)->create();

         $this->actingAs($this->user)
            ->from(route($this->indexRoute))
            ->post(route($this->bulkRoute, ['bulk-action' => 'inactive', 'id' => $users->pluck('id')->all()]))
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('user.inactive_success'));
    }

    /** @test */
    public function testBulkActionDelete()
    {
        $users = factory(User::class, 2)->create();

         $this->actingAs($this->user)
            ->from(route($this->indexRoute))
            ->post(route($this->bulkRoute, ['bulk-action' => 'delete', 'id' => $users->pluck('id')->all()]))
            ->assertStatus(302)
            ->assertRedirect(route($this->indexRoute))
            ->assertSessionHas('success', __('user.delete_success'));
    }
}
